// Merged JS - generated

/* --- assets/js/admin.js --- */
// assets/js/admin.js - Admin Panel JavaScript
// جاوااسکریپت پنل مدیریت

class AdminManager {
    constructor() {
        this.currentPage = 'dashboard';
        this.charts = {};
        this.tables = {};
        this.filters = {};
        this.isLoading = false;
        
        this.init();
    }
    
    init() {
        this.setupEventListeners();
        this.loadDashboardData();
        this.initializeCharts();
        this.setupDataTables();
        this.startRealTimeUpdates();
    }
    
    // Setup event listeners
    setupEventListeners() {
        // Sidebar navigation
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('admin-nav-link')) {
                e.preventDefault();
                this.navigateToPage(e.target.dataset.page);
            }
            
            // User actions
            if (e.target.classList.contains('ban-user-btn')) {
                this.banUser(e.target.dataset.userId);
            }
            
            if (e.target.classList.contains('unban-user-btn')) {
                this.unbanUser(e.target.dataset.userId);
            }
            
            if (e.target.classList.contains('verify-user-btn')) {
                this.verifyUser(e.target.dataset.userId);
            }
            
            // Transaction actions
            if (e.target.classList.contains('approve-transaction-btn')) {
                this.approveTransaction(e.target.dataset.transactionId);
            }
            
            if (e.target.classList.contains('reject-transaction-btn')) {
                this.rejectTransaction(e.target.dataset.transactionId);
            }
            
            // Support ticket actions
            if (e.target.classList.contains('reply-ticket-btn')) {
                this.showTicketReplyModal(e.target.dataset.ticketId);
            }
            
            if (e.target.classList.contains('close-ticket-btn')) {
                this.closeTicket(e.target.dataset.ticketId);
            }
            
            // Settings actions
            if (e.target.classList.contains('save-settings-btn')) {
                this.saveSettings();
            }
            
            // Export actions
            if (e.target.classList.contains('export-btn')) {
                this.exportData(e.target.dataset.type);
            }
        });
        
        // Form submissions
        document.addEventListener('submit', (e) => {
            if (e.target.classList.contains('admin-form')) {
                e.preventDefault();
                this.handleFormSubmission(e.target);
            }
        });
        
        // Filter changes
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('admin-filter')) {
                this.applyFilters();
            }
        });
        
        // Search functionality
        document.addEventListener('input', (e) => {
            if (e.target.classList.contains('admin-search')) {
                this.debounce(() => {
                    this.performSearch(e.target.value, e.target.dataset.searchType);
                }, 500)();
            }
        });
    }
    
    // Navigate to different admin pages
    navigateToPage(page) {
        // Update active navigation
        document.querySelectorAll('.admin-nav-link').forEach(link => {
            link.classList.remove('active');
        });
        
        document.querySelector(`[data-page="${page}"]`).classList.add('active');
        
        // Hide all page contents
        document.querySelectorAll('.admin-page-content').forEach(content => {
            content.style.display = 'none';
        });
        
        // Show selected page
        const pageContent = document.querySelector(`#admin-${page}`);
        if (pageContent) {
            pageContent.style.display = 'block';
            this.currentPage = page;
            
            // Load page-specific data
            this.loadPageData(page);
        }
    }
    
    // Load dashboard data
    async loadDashboardData() {
        try {
            this.showLoading();
            
            const response = await fetch('/admin/api/dashboard.php');
            const data = await response.json();
            
            if (data.success) {
                this.updateDashboardStats(data.data.stats);
                this.updateDashboardCharts(data.data.charts);
                this.updateRecentActivity(data.data.recent_activity);
            } else {
                throw new Error(data.error || 'خطا در بارگذاری داده‌های داشبورد');
            }
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showNotification(error.message, 'error');
        } finally {
            this.hideLoading();
        }
    }
    
    // Update dashboard statistics
    updateDashboardStats(stats) {
        Object.keys(stats).forEach(key => {
            const element = document.querySelector(`[data-stat="${key}"]`);
            if (element) {
                const valueElement = element.querySelector('.stat-value');
                const changeElement = element.querySelector('.stat-change');
                
                if (valueElement) {
                    this.animateNumber(valueElement, stats[key].value);
                }
                
                if (changeElement && stats[key].change !== undefined) {
                    const change = stats[key].change;
                    changeElement.textContent = `${change > 0 ? '+' : ''}${change}%`;
                    changeElement.className = `stat-change ${change >= 0 ? 'positive' : 'negative'}`;
                }
            }
        });
    }
    
    // User management actions
    async banUser(userId) {
        if (!confirm('آیا از مسدود کردن این کاربر اطمینان دارید؟')) {
            return;
        }
        
        try {
            const response = await fetch('/admin/api/users.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'ban',
                    user_id: userId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification('کاربر با موفقیت مسدود شد', 'success');
                this.loadUsersData();
            } else {
                throw new Error(result.error || 'خطا در مسدود کردن کاربر');
            }
        } catch (error) {
            console.error('Error banning user:', error);
            this.showNotification(error.message, 'error');
        }
    }
    
    async unbanUser(userId) {
        try {
            const response = await fetch('/admin/api/users.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'unban',
                    user_id: userId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification('کاربر با موفقیت فعال شد', 'success');
                this.loadUsersData();
            } else {
                throw new Error(result.error || 'خطا در فعال کردن کاربر');
            }
        } catch (error) {
            console.error('Error unbanning user:', error);
            this.showNotification(error.message, 'error');
        }
    }
    
    // Initialize charts
    initializeCharts() {
        // Revenue chart
        const revenueCtx = document.getElementById('revenueChart');
        if (revenueCtx) {
            this.charts.revenue = new Chart(revenueCtx, {
                type: 'line',
                data: {
                    labels: [],
                    datasets: [{
                        label: 'درآمد',
                        data: [],
                        borderColor: '#007bff',
                        backgroundColor: 'rgba(0, 123, 255, 0.1)',
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }
        
        // Users chart
        const usersCtx = document.getElementById('usersChart');
        if (usersCtx) {
            this.charts.users = new Chart(usersCtx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        label: 'کاربران جدید',
                        data: [],
                        backgroundColor: '#28a745'
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }
    }
    
    // Utility functions
    animateNumber(element, targetValue) {
        const startValue = parseInt(element.textContent) || 0;
        const duration = 1000;
        const startTime = Date.now();
        
        const animate = () => {
            const elapsed = Date.now() - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            const currentValue = Math.floor(startValue + (targetValue - startValue) * progress);
            element.textContent = currentValue.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        animate();
    }
    
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">&times;</button>
        `;
        
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        notification.querySelector('.notification-close').addEventListener('click', () => {
            this.removeNotification(notification);
        });
        
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);
    }
    
    removeNotification(notification) {
        if (notification.parentNode) {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }
    }
    
    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'exclamation-circle',
            'warning': 'exclamation-triangle',
            'info': 'info-circle'
        };
        
        return icons[type] || 'info-circle';
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.admin-loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'admin-loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="loading-spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        }
        
        loader.style.display = 'flex';
        this.isLoading = true;
    }
    
    hideLoading() {
        const loader = document.querySelector('.admin-loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
        this.isLoading = false;
    }
    
    startRealTimeUpdates() {
        setInterval(() => {
            if (this.currentPage === 'dashboard') {
                this.loadDashboardData();
            }
        }, 30000);
    }
    
    setupDataTables() {
        // Initialize DataTables for admin tables
        if (typeof $ !== 'undefined' && $.fn.DataTable) {
            $('.admin-table').DataTable({
                responsive: true,
                language: {
                    url: '/assets/js/datatables-fa.json'
                }
            });
        }
    }
}

// Initialize admin manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    if (document.body.classList.contains('admin-page')) {
        window.adminManager = new AdminManager();
    }
});

/* --- assets/js/betting.js --- */
// assets/js/betting.js - Enhanced Betting JavaScript
// جاوااسکریپت پیشرفته شرط‌بندی

class BettingManager {
    constructor() {
        this.activeBets = new Map();
        this.betHistory = [];
        this.balance = 0;
        this.minBet = 1;
        this.maxBet = 10000;
        this.isConnected = false;
        this.eventListeners = new Map();
        
        this.init();
    }
    
    init() {
        this.loadUserData();
        this.setupEventListeners();
        this.connectWebSocket();
        this.startPeriodicUpdates();
    }
    
    // Load user data
    async loadUserData() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=balance');
            const data = await response.json();
            
            if (data.success) {
                this.balance = data.data.balance;
                this.updateBalanceDisplay();
            }
        } catch (error) {
            console.error('Error loading user data:', error);
            this.showNotification('خطا در بارگذاری اطلاعات کاربر', 'error');
        }
    }
    
    // Setup event listeners
    setupEventListeners() {
        // Bet placement
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('place-bet-btn')) {
                this.handleBetPlacement(e.target);
            }
            
            if (e.target.classList.contains('cashout-btn')) {
                this.handleCashout(e.target);
            }
            
            if (e.target.classList.contains('cancel-bet-btn')) {
                this.handleBetCancellation(e.target);
            }
        });
        
        // Bet amount validation
        document.addEventListener('input', (e) => {
            if (e.target.classList.contains('bet-amount-input')) {
                this.validateBetAmount(e.target);
            }
        });
        
        // Quick bet buttons
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('quick-bet-btn')) {
                this.handleQuickBet(e.target);
            }
        });
    }
    
    // WebSocket connection for real-time updates
    connectWebSocket() {
        try {
            this.ws = new WebSocket('wss://crashify.com/ws');
            
            this.ws.onopen = () => {
                this.isConnected = true;
                console.log('WebSocket connected');
                this.showNotification('اتصال برقرار شد', 'success');
            };
            
            this.ws.onmessage = (event) => {
                const data = JSON.parse(event.data);
                this.handleWebSocketMessage(data);
            };
            
            this.ws.onclose = () => {
                this.isConnected = false;
                console.log('WebSocket disconnected');
                this.showNotification('اتصال قطع شد', 'warning');
                
                // Reconnect after 5 seconds
                setTimeout(() => this.connectWebSocket(), 5000);
            };
            
            this.ws.onerror = (error) => {
                console.error('WebSocket error:', error);
                this.showNotification('خطا در اتصال', 'error');
            };
        } catch (error) {
            console.error('Failed to connect WebSocket:', error);
        }
    }
    
    // Handle WebSocket messages
    handleWebSocketMessage(data) {
        switch (data.type) {
            case 'odds_update':
                this.updateOdds(data.odds);
                break;
            case 'match_update':
                this.updateMatchStatus(data.match);
                break;
            case 'bet_result':
                this.handleBetResult(data.result);
                break;
            case 'balance_update':
                this.updateBalance(data.balance);
                break;
            case 'crash_update':
                this.updateCrashMultiplier(data.multiplier);
                break;
        }
    }
    
    // Place a bet
    async placeBet(gameType, betData) {
        try {
            // Validate bet amount
            if (betData.bet_amount < this.minBet || betData.bet_amount > this.maxBet) {
                throw new Error(`مبلغ شرط باید بین ${this.minBet} تا ${this.maxBet} باشد`);
            }
            
            if (betData.bet_amount > this.balance) {
                throw new Error('موجودی کافی نیست');
            }
            
            // Show loading
            this.showLoading('در حال ثبت شرط...');
            
            const response = await fetch('/api/bets.php?endpoint=place', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    game_type: gameType,
                    bet_amount: betData.bet_amount,
                    bet_data: betData
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Add to active bets
                this.activeBets.set(result.data.bet_id, {
                    ...result.data,
                    timestamp: Date.now()
                });
                
                // Update balance
                this.balance -= betData.bet_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification('شرط با موفقیت ثبت شد', 'success');
                
                // Trigger custom event
                this.triggerEvent('betPlaced', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در ثبت شرط');
            }
        } catch (error) {
            console.error('Error placing bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle bet placement from UI
    async handleBetPlacement(button) {
        const betForm = button.closest('.bet-form');
        if (!betForm) return;
        
        const gameType = betForm.dataset.gameType;
        const betAmount = parseFloat(betForm.querySelector('.bet-amount-input').value);
        const betData = this.extractBetData(betForm);
        
        try {
            await this.placeBet(gameType, {
                bet_amount: betAmount,
                ...betData
            });
        } catch (error) {
            // Error already handled in placeBet
        }
    }
    
    // Extract bet data from form
    extractBetData(form) {
        const data = {};
        
        // Get all form inputs
        const inputs = form.querySelectorAll('input, select');
        inputs.forEach(input => {
            if (input.name && input.value) {
                data[input.name] = input.value;
            }
        });
        
        // Get selected odds
        const selectedOdds = form.querySelector('.odds-button.selected');
        if (selectedOdds) {
            data.odds = parseFloat(selectedOdds.dataset.odds);
            data.bet_type = selectedOdds.dataset.betType;
            data.match_id = selectedOdds.dataset.matchId;
        }
        
        return data;
    }
    
    // Cashout bet
    async cashoutBet(betId, multiplier = null) {
        try {
            this.showLoading('در حال تسویه...');
            
            const response = await fetch('/api/bets.php?endpoint=cashout', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    bet_id: betId,
                    multiplier: multiplier
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Remove from active bets
                this.activeBets.delete(betId);
                
                // Update balance
                this.balance += result.data.payout_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification(`تسویه موفق! برد: ${result.data.profit}`, 'success');
                
                // Trigger custom event
                this.triggerEvent('betCashedOut', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در تسویه');
            }
        } catch (error) {
            console.error('Error cashing out bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle cashout from UI
    async handleCashout(button) {
        const betId = parseInt(button.dataset.betId);
        const multiplier = button.dataset.multiplier ? parseFloat(button.dataset.multiplier) : null;
        
        try {
            await this.cashoutBet(betId, multiplier);
        } catch (error) {
            // Error already handled in cashoutBet
        }
    }
    
    // Cancel bet
    async cancelBet(betId) {
        try {
            this.showLoading('در حال لغو شرط...');
            
            const response = await fetch('/api/bets.php?endpoint=cancel', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    bet_id: betId
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Remove from active bets
                this.activeBets.delete(betId);
                
                // Update balance
                this.balance += result.data.refund_amount;
                this.updateBalanceDisplay();
                
                // Update UI
                this.updateActiveBetsDisplay();
                this.showNotification('شرط لغو شد', 'info');
                
                // Trigger custom event
                this.triggerEvent('betCancelled', result.data);
                
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در لغو شرط');
            }
        } catch (error) {
            console.error('Error cancelling bet:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Handle bet cancellation from UI
    async handleBetCancellation(button) {
        const betId = parseInt(button.dataset.betId);
        
        if (confirm('آیا از لغو این شرط اطمینان دارید؟')) {
            try {
                await this.cancelBet(betId);
            } catch (error) {
                // Error already handled in cancelBet
            }
        }
    }
    
    // Quick bet functionality
    handleQuickBet(button) {
        const amount = parseFloat(button.dataset.amount);
        const betInput = button.closest('.bet-controls').querySelector('.bet-amount-input');
        
        if (betInput) {
            if (button.dataset.action === 'set') {
                betInput.value = amount;
            } else if (button.dataset.action === 'add') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount + amount;
            } else if (button.dataset.action === 'multiply') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount * amount;
            } else if (button.dataset.action === 'max') {
                betInput.value = Math.min(this.balance, this.maxBet);
            } else if (button.dataset.action === 'half') {
                const currentAmount = parseFloat(betInput.value) || 0;
                betInput.value = currentAmount / 2;
            }
            
            this.validateBetAmount(betInput);
        }
    }
    
    // Validate bet amount
    validateBetAmount(input) {
        const amount = parseFloat(input.value);
        const errorElement = input.parentElement.querySelector('.bet-error');
        
        let isValid = true;
        let errorMessage = '';
        
        if (isNaN(amount) || amount <= 0) {
            isValid = false;
            errorMessage = 'مبلغ نامعتبر';
        } else if (amount < this.minBet) {
            isValid = false;
            errorMessage = `حداقل مبلغ: ${this.minBet}`;
        } else if (amount > this.maxBet) {
            isValid = false;
            errorMessage = `حداکثر مبلغ: ${this.maxBet}`;
        } else if (amount > this.balance) {
            isValid = false;
            errorMessage = 'موجودی کافی نیست';
        }
        
        // Update UI
        input.classList.toggle('is-invalid', !isValid);
        input.classList.toggle('is-valid', isValid);
        
        if (errorElement) {
            errorElement.textContent = errorMessage;
            errorElement.style.display = errorMessage ? 'block' : 'none';
        }
        
        // Enable/disable bet button
        const betButton = input.closest('.bet-form').querySelector('.place-bet-btn');
        if (betButton) {
            betButton.disabled = !isValid;
        }
        
        return isValid;
    }
    
    // Update odds display
    updateOdds(oddsData) {
        oddsData.forEach(odd => {
            const oddsButton = document.querySelector(`[data-match-id="${odd.match_id}"][data-bet-type="${odd.bet_type}"]`);
            if (oddsButton) {
                const oldOdds = parseFloat(oddsButton.dataset.odds);
                const newOdds = odd.odds;
                
                oddsButton.dataset.odds = newOdds;
                oddsButton.querySelector('.odds-value').textContent = newOdds.toFixed(2);
                
                // Show change indicator
                if (newOdds > oldOdds) {
                    oddsButton.classList.add('odds-up');
                    setTimeout(() => oddsButton.classList.remove('odds-up'), 2000);
                } else if (newOdds < oldOdds) {
                    oddsButton.classList.add('odds-down');
                    setTimeout(() => oddsButton.classList.remove('odds-down'), 2000);
                }
            }
        });
    }
    
    // Update match status
    updateMatchStatus(matchData) {
        const matchElement = document.querySelector(`[data-match-id="${matchData.id}"]`);
        if (matchElement) {
            const statusElement = matchElement.querySelector('.match-status');
            const scoreElement = matchElement.querySelector('.match-score');
            const timeElement = matchElement.querySelector('.match-time');
            
            if (statusElement) statusElement.textContent = matchData.status;
            if (scoreElement) scoreElement.textContent = matchData.score;
            if (timeElement) timeElement.textContent = matchData.time;
        }
    }
    
    // Handle bet result
    handleBetResult(resultData) {
        const bet = this.activeBets.get(resultData.bet_id);
        if (bet) {
            this.activeBets.delete(resultData.bet_id);
            
            if (resultData.status === 'won') {
                this.balance += resultData.payout_amount;
                this.showNotification(`تبریک! شما برنده شدید: ${resultData.profit}`, 'success');
            } else if (resultData.status === 'lost') {
                this.showNotification('متاسفانه شرط شما باخت', 'error');
            }
            
            this.updateBalanceDisplay();
            this.updateActiveBetsDisplay();
            
            // Trigger custom event
            this.triggerEvent('betResult', resultData);
        }
    }
    
    // Update balance
    updateBalance(newBalance) {
        this.balance = newBalance;
        this.updateBalanceDisplay();
    }
    
    // Update crash multiplier
    updateCrashMultiplier(multiplier) {
        const multiplierElements = document.querySelectorAll('.crash-multiplier');
        multiplierElements.forEach(element => {
            element.textContent = multiplier.toFixed(2) + 'x';
        });
        
        // Update potential payout for active crash bets
        this.activeBets.forEach((bet, betId) => {
            if (bet.game_type === 'crash' && bet.status === 'active') {
                const potentialPayout = bet.bet_amount * multiplier;
                const payoutElement = document.querySelector(`[data-bet-id="${betId}"] .potential-payout`);
                if (payoutElement) {
                    payoutElement.textContent = potentialPayout.toFixed(2);
                }
            }
        });
    }
    
    // Update balance display
    updateBalanceDisplay() {
        const balanceElements = document.querySelectorAll('.user-balance');
        balanceElements.forEach(element => {
            element.textContent = this.formatCurrency(this.balance);
        });
    }
    
    // Update active bets display
    updateActiveBetsDisplay() {
        const container = document.querySelector('.active-bets-container');
        if (!container) return;
        
        if (this.activeBets.size === 0) {
            container.innerHTML = '<p class="text-muted text-center">شرط فعالی وجود ندارد</p>';
            return;
        }
        
        let html = '';
        this.activeBets.forEach((bet, betId) => {
            html += this.generateActiveBetHTML(bet, betId);
        });
        
        container.innerHTML = html;
    }
    
    // Generate HTML for active bet
    generateActiveBetHTML(bet, betId) {
        return `
            <div class="active-bet-item" data-bet-id="${betId}">
                <div class="bet-info">
                    <span class="bet-game">${bet.game_type}</span>
                    <span class="bet-amount">${this.formatCurrency(bet.bet_amount)}</span>
                </div>
                <div class="bet-actions">
                    ${bet.game_type === 'crash' ? 
                        `<button class="btn btn-sm btn-success cashout-btn" data-bet-id="${betId}">
                            تسویه
                        </button>` : 
                        `<button class="btn btn-sm btn-danger cancel-bet-btn" data-bet-id="${betId}">
                            لغو
                        </button>`
                    }
                </div>
            </div>
        `;
    }
    
    // Load bet history
    async loadBetHistory(page = 1, filters = {}) {
        try {
            const params = new URLSearchParams({
                page: page,
                limit: 20,
                ...filters
            });
            
            const response = await fetch(`/api/bets.php?endpoint=history&${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.betHistory = data.data.bets;
                this.updateBetHistoryDisplay(data.data);
                return data.data;
            } else {
                throw new Error(data.error || 'خطا در بارگذاری تاریخچه');
            }
        } catch (error) {
            console.error('Error loading bet history:', error);
            this.showNotification(error.message, 'error');
            throw error;
        }
    }
    
    // Update bet history display
    updateBetHistoryDisplay(historyData) {
        const container = document.querySelector('.bet-history-container');
        if (!container) return;
        
        let html = '';
        historyData.bets.forEach(bet => {
            html += this.generateBetHistoryHTML(bet);
        });
        
        container.innerHTML = html;
        
        // Update pagination
        this.updatePagination(historyData.pagination);
    }
    
    // Generate HTML for bet history item
    generateBetHistoryHTML(bet) {
        const statusClass = {
            'won': 'success',
            'lost': 'danger',
            'pending': 'warning',
            'cancelled': 'secondary'
        }[bet.status] || 'secondary';
        
        return `
            <div class="bet-history-item">
                <div class="bet-details">
                    <div class="bet-game">${bet.game_type}</div>
                    <div class="bet-amount">${this.formatCurrency(bet.bet_amount)}</div>
                    <div class="bet-date">${this.formatDate(bet.created_at)}</div>
                </div>
                <div class="bet-result">
                    <span class="badge badge-${statusClass}">${bet.status}</span>
                    ${bet.payout_amount ? 
                        `<div class="payout">${this.formatCurrency(bet.payout_amount)}</div>` : 
                        ''
                    }
                </div>
            </div>
        `;
    }
    
    // Periodic updates
    startPeriodicUpdates() {
        // Update active bets every 5 seconds
        setInterval(() => {
            if (this.activeBets.size > 0) {
                this.updateActiveBets();
            }
        }, 5000);
        
        // Update balance every 30 seconds
        setInterval(() => {
            this.loadUserData();
        }, 30000);
    }
    
    // Update active bets from server
    async updateActiveBets() {
        try {
            const response = await fetch('/api/bets.php?endpoint=active');
            const data = await response.json();
            
            if (data.success) {
                // Update local active bets
                this.activeBets.clear();
                data.data.active_bets.forEach(bet => {
                    this.activeBets.set(bet.id, bet);
                });
                
                this.updateActiveBetsDisplay();
            }
        } catch (error) {
            console.error('Error updating active bets:', error);
        }
    }
    
    // Event system
    addEventListener(event, callback) {
        if (!this.eventListeners.has(event)) {
            this.eventListeners.set(event, []);
        }
        this.eventListeners.get(event).push(callback);
    }
    
    removeEventListener(event, callback) {
        if (this.eventListeners.has(event)) {
            const listeners = this.eventListeners.get(event);
            const index = listeners.indexOf(callback);
            if (index > -1) {
                listeners.splice(index, 1);
            }
        }
    }
    
    triggerEvent(event, data) {
        if (this.eventListeners.has(event)) {
            this.eventListeners.get(event).forEach(callback => {
                try {
                    callback(data);
                } catch (error) {
                    console.error('Error in event listener:', error);
                }
            });
        }
    }
    
    // Utility functions
    formatCurrency(amount) {
        return new Intl.NumberFormat('fa-IR', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 2
        }).format(amount);
    }
    
    formatDate(dateString) {
        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        }).format(new Date(dateString));
    }
    
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.textContent = message;
        
        // Add to container
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 5000);
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        } else {
            loader.querySelector('.loading-message').textContent = message;
        }
        
        loader.style.display = 'flex';
    }
    
    hideLoading() {
        const loader = document.querySelector('.loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
    }
    
    updatePagination(paginationData) {
        const container = document.querySelector('.pagination-container');
        if (!container) return;
        
        let html = '<nav><ul class="pagination">';
        
        // Previous button
        if (paginationData.current_page > 1) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page - 1}">قبلی</a>
            </li>`;
        }
        
        // Page numbers
        for (let i = 1; i <= paginationData.total_pages; i++) {
            const isActive = i === paginationData.current_page;
            html += `<li class="page-item ${isActive ? 'active' : ''}">
                <a class="page-link" href="#" data-page="${i}">${i}</a>
            </li>`;
        }
        
        // Next button
        if (paginationData.current_page < paginationData.total_pages) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page + 1}">بعدی</a>
            </li>`;
        }
        
        html += '</ul></nav>';
        container.innerHTML = html;
        
        // Add click handlers
        container.querySelectorAll('.page-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = parseInt(e.target.dataset.page);
                this.loadBetHistory(page);
            });
        });
    }
}

// Initialize betting manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.bettingManager = new BettingManager();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = BettingManager;
}
/* --- assets/js/casino.js --- */
// assets/js/casino.js - Casino Games JavaScript
// جاوااسکریپت بازی‌های کازینو

class CasinoManager {
    constructor() {
        this.games = new Map();
        this.activeGame = null;
        this.gameHistory = [];
        this.balance = 0;
        this.isPlaying = false;
        
        this.init();
    }
    
    init() {
        this.loadGames();
        this.setupEventListeners();
        this.loadBalance();
    }
    
    // Load available games
    async loadGames() {
        try {
            // Mock games data - in production, load from API
            const games = [
                {
                    id: 'slots',
                    name: 'اسلات',
                    type: 'slots',
                    minBet: 0.1,
                    maxBet: 100,
                    rtp: 96.5,
                    volatility: 'medium'
                },
                {
                    id: 'blackjack',
                    name: 'بلک جک',
                    type: 'table',
                    minBet: 1,
                    maxBet: 500,
                    rtp: 99.5,
                    volatility: 'low'
                },
                {
                    id: 'roulette',
                    name: 'رولت',
                    type: 'table',
                    minBet: 0.5,
                    maxBet: 1000,
                    rtp: 97.3,
                    volatility: 'medium'
                },
                {
                    id: 'baccarat',
                    name: 'باکارات',
                    type: 'table',
                    minBet: 2,
                    maxBet: 2000,
                    rtp: 98.9,
                    volatility: 'low'
                }
            ];
            
            games.forEach(game => {
                this.games.set(game.id, game);
            });
            
            this.renderGamesGrid();
        } catch (error) {
            console.error('Error loading games:', error);
        }
    }
    
    // Setup event listeners
    setupEventListeners() {
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('play-game-btn')) {
                this.startGame(e.target.dataset.gameId);
            }
            
            if (e.target.classList.contains('spin-btn')) {
                this.spinSlots();
            }
            
            if (e.target.classList.contains('deal-btn')) {
                this.dealCards();
            }
            
            if (e.target.classList.contains('hit-btn')) {
                this.hitCard();
            }
            
            if (e.target.classList.contains('stand-btn')) {
                this.standCards();
            }
            
            if (e.target.classList.contains('bet-red-btn')) {
                this.placeBet('red');
            }
            
            if (e.target.classList.contains('bet-black-btn')) {
                this.placeBet('black');
            }
            
            if (e.target.classList.contains('bet-number-btn')) {
                this.placeBet('number', e.target.dataset.number);
            }
        });
    }
    
    // Load user balance
    async loadBalance() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=balance');
            const data = await response.json();
            
            if (data.success) {
                this.balance = data.data.balance;
                this.updateBalanceDisplay();
            }
        } catch (error) {
            console.error('Error loading balance:', error);
        }
    }
    
    // Render games grid
    renderGamesGrid() {
        const container = document.querySelector('.games-grid');
        if (!container) return;
        
        let html = '';
        this.games.forEach((game, gameId) => {
            html += `
                <div class="game-card" data-game-id="${gameId}">
                    <div class="game-image">
                        <img src="/assets/images/games/${gameId}.jpg" alt="${game.name}" 
                             onerror="this.src='/assets/images/games/default.jpg'">
                    </div>
                    <div class="game-info">
                        <h3 class="game-title">${game.name}</h3>
                        <div class="game-stats">
                            <span class="rtp">RTP: ${game.rtp}%</span>
                            <span class="volatility">${game.volatility}</span>
                        </div>
                        <div class="game-limits">
                            <span>حداقل: $${game.minBet}</span>
                            <span>حداکثر: $${game.maxBet}</span>
                        </div>
                        <button class="btn btn-primary play-game-btn" data-game-id="${gameId}">
                            بازی کنید
                        </button>
                    </div>
                </div>
            `;
        });
        
        container.innerHTML = html;
    }
    
    // Start a game
    async startGame(gameId) {
        const game = this.games.get(gameId);
        if (!game) return;
        
        this.activeGame = game;
        
        // Show game interface
        this.showGameInterface(game);
        
        // Initialize game state
        switch (game.type) {
            case 'slots':
                this.initSlots();
                break;
            case 'table':
                if (gameId === 'blackjack') {
                    this.initBlackjack();
                } else if (gameId === 'roulette') {
                    this.initRoulette();
                } else if (gameId === 'baccarat') {
                    this.initBaccarat();
                }
                break;
        }
    }
    
    // Show game interface
    showGameInterface(game) {
        const modal = document.createElement('div');
        modal.className = 'game-modal';
        modal.innerHTML = this.getGameHTML(game);
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelector('.close-game').addEventListener('click', () => {
            this.closeGame();
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Get game HTML
    getGameHTML(game) {
        switch (game.id) {
            case 'slots':
                return this.getSlotsHTML();
            case 'blackjack':
                return this.getBlackjackHTML();
            case 'roulette':
                return this.getRouletteHTML();
            case 'baccarat':
                return this.getBaccaratHTML();
            default:
                return '<div>Game not implemented</div>';
        }
    }
    
    // Slots game
    getSlotsHTML() {
        return `
            <div class="game-container slots-game">
                <div class="game-header">
                    <h2>اسلات ماشین</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="slots-machine">
                        <div class="reels">
                            <div class="reel" id="reel1">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                            <div class="reel" id="reel2">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                            <div class="reel" id="reel3">
                                <div class="symbol">🍒</div>
                                <div class="symbol">🍋</div>
                                <div class="symbol">🍊</div>
                            </div>
                        </div>
                        <div class="paylines"></div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="0.1" max="100" value="1" step="0.1">
                        </div>
                        <button class="btn btn-primary spin-btn" ${this.isPlaying ? 'disabled' : ''}>
                            چرخاندن
                        </button>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="last-win">آخرین برد: $<span class="last-win-amount">0</span></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Initialize slots
    initSlots() {
        this.slotsSymbols = ['🍒', '🍋', '🍊', '🍇', '🍎', '💎', '⭐', '🔔'];
        this.slotsPaytable = {
            '🍒🍒🍒': 10,
            '🍋🍋🍋': 15,
            '🍊🍊🍊': 20,
            '🍇🍇🍇': 25,
            '🍎🍎🍎': 30,
            '💎💎💎': 100,
            '⭐⭐⭐': 200,
            '🔔🔔🔔': 500
        };
    }
    
    // Spin slots
    async spinSlots() {
        if (this.isPlaying) return;
        
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (betAmount > this.balance) {
            this.showNotification('موجودی کافی نیست', 'error');
            return;
        }
        
        this.isPlaying = true;
        document.querySelector('.spin-btn').disabled = true;
        
        // Deduct bet amount
        this.balance -= betAmount;
        this.updateBalanceDisplay();
        
        // Animate reels
        const reels = document.querySelectorAll('.reel');
        reels.forEach((reel, index) => {
            this.animateReel(reel, index * 500);
        });
        
        // Wait for animation to complete
        setTimeout(() => {
            const result = this.getSpinResult();
            this.processSpinResult(result, betAmount);
            
            this.isPlaying = false;
            document.querySelector('.spin-btn').disabled = false;
        }, 2000);
    }
    
    // Animate reel
    animateReel(reel, delay) {
        setTimeout(() => {
            reel.classList.add('spinning');
            
            setTimeout(() => {
                // Set final symbol
                const randomSymbol = this.slotsSymbols[Math.floor(Math.random() * this.slotsSymbols.length)];
                reel.querySelector('.symbol').textContent = randomSymbol;
                reel.classList.remove('spinning');
            }, 1500);
        }, delay);
    }
    
    // Get spin result
    getSpinResult() {
        const reels = document.querySelectorAll('.reel .symbol');
        return Array.from(reels).map(symbol => symbol.textContent);
    }
    
    // Process spin result
    processSpinResult(result, betAmount) {
        const combination = result.join('');
        const payout = this.slotsPaytable[combination] || 0;
        const winAmount = payout * betAmount;
        
        if (winAmount > 0) {
            this.balance += winAmount;
            this.updateBalanceDisplay();
            document.querySelector('.last-win-amount').textContent = winAmount.toFixed(2);
            this.showNotification(`تبریک! شما $${winAmount.toFixed(2)} برنده شدید!`, 'success');
            
            // Highlight winning combination
            document.querySelectorAll('.reel').forEach(reel => {
                reel.classList.add('winner');
            });
            
            setTimeout(() => {
                document.querySelectorAll('.reel').forEach(reel => {
                    reel.classList.remove('winner');
                });
            }, 3000);
        }
        
        // Record game result
        this.recordGameResult('slots', betAmount, winAmount);
    }
    
    // Blackjack game
    getBlackjackHTML() {
        return `
            <div class="game-container blackjack-game">
                <div class="game-header">
                    <h2>بلک جک</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="dealer-area">
                        <h3>دیلر</h3>
                        <div class="dealer-cards"></div>
                        <div class="dealer-score">امتیاز: <span>0</span></div>
                    </div>
                    <div class="player-area">
                        <h3>شما</h3>
                        <div class="player-cards"></div>
                        <div class="player-score">امتیاز: <span>0</span></div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="1" max="500" value="10" step="1">
                            <button class="btn btn-primary deal-btn">شروع بازی</button>
                        </div>
                        <div class="play-controls" style="display: none;">
                            <button class="btn btn-success hit-btn">کارت بگیر</button>
                            <button class="btn btn-warning stand-btn">بایست</button>
                        </div>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="game-status"></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Initialize blackjack
    initBlackjack() {
        this.deck = this.createDeck();
        this.playerCards = [];
        this.dealerCards = [];
        this.gameState = 'betting'; // betting, playing, finished
    }
    
    // Create deck of cards
    createDeck() {
        const suits = ['♠', '♥', '♦', '♣'];
        const ranks = ['A', '2', '3', '4', '5', '6', '7', '8', '9', '10', 'J', 'Q', 'K'];
        const deck = [];
        
        suits.forEach(suit => {
            ranks.forEach(rank => {
                deck.push({ suit, rank, value: this.getCardValue(rank) });
            });
        });
        
        return this.shuffleDeck(deck);
    }
    
    // Get card value for blackjack
    getCardValue(rank) {
        if (rank === 'A') return 11;
        if (['J', 'Q', 'K'].includes(rank)) return 10;
        return parseInt(rank);
    }
    
    // Shuffle deck
    shuffleDeck(deck) {
        for (let i = deck.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [deck[i], deck[j]] = [deck[j], deck[i]];
        }
        return deck;
    }
    
    // Deal initial cards
    dealCards() {
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (betAmount > this.balance) {
            this.showNotification('موجودی کافی نیست', 'error');
            return;
        }
        
        // Deduct bet amount
        this.balance -= betAmount;
        this.updateBalanceDisplay();
        
        // Reset game
        this.playerCards = [];
        this.dealerCards = [];
        
        // Deal initial cards
        this.playerCards.push(this.drawCard());
        this.dealerCards.push(this.drawCard());
        this.playerCards.push(this.drawCard());
        this.dealerCards.push(this.drawCard());
        
        this.updateBlackjackDisplay();
        
        // Check for blackjack
        if (this.calculateScore(this.playerCards) === 21) {
            this.endBlackjackGame('blackjack', betAmount);
        } else {
            this.gameState = 'playing';
            document.querySelector('.bet-controls').style.display = 'none';
            document.querySelector('.play-controls').style.display = 'block';
        }
    }
    
    // Draw a card from deck
    drawCard() {
        if (this.deck.length === 0) {
            this.deck = this.createDeck();
        }
        return this.deck.pop();
    }
    
    // Hit (take another card)
    hitCard() {
        if (this.gameState !== 'playing') return;
        
        this.playerCards.push(this.drawCard());
        this.updateBlackjackDisplay();
        
        const playerScore = this.calculateScore(this.playerCards);
        if (playerScore > 21) {
            this.endBlackjackGame('bust', parseFloat(document.querySelector('.bet-amount').value));
        }
    }
    
    // Stand (end turn)
    standCards() {
        if (this.gameState !== 'playing') return;
        
        // Dealer plays
        while (this.calculateScore(this.dealerCards) < 17) {
            this.dealerCards.push(this.drawCard());
        }
        
        this.updateBlackjackDisplay();
        
        const playerScore = this.calculateScore(this.playerCards);
        const dealerScore = this.calculateScore(this.dealerCards);
        const betAmount = parseFloat(document.querySelector('.bet-amount').value);
        
        if (dealerScore > 21) {
            this.endBlackjackGame('dealer_bust', betAmount);
        } else if (playerScore > dealerScore) {
            this.endBlackjackGame('win', betAmount);
        } else if (playerScore < dealerScore) {
            this.endBlackjackGame('lose', betAmount);
        } else {
            this.endBlackjackGame('push', betAmount);
        }
    }
    
    // Calculate blackjack score
    calculateScore(cards) {
        let score = 0;
        let aces = 0;
        
        cards.forEach(card => {
            if (card.rank === 'A') {
                aces++;
                score += 11;
            } else {
                score += card.value;
            }
        });
        
        // Adjust for aces
        while (score > 21 && aces > 0) {
            score -= 10;
            aces--;
        }
        
        return score;
    }
    
    // Update blackjack display
    updateBlackjackDisplay() {
        const playerCardsEl = document.querySelector('.player-cards');
        const dealerCardsEl = document.querySelector('.dealer-cards');
        const playerScoreEl = document.querySelector('.player-score span');
        const dealerScoreEl = document.querySelector('.dealer-score span');
        
        // Show player cards
        playerCardsEl.innerHTML = this.playerCards.map(card => 
            `<div class="card">${card.rank}${card.suit}</div>`
        ).join('');
        
        // Show dealer cards (hide second card if game is still playing)
        if (this.gameState === 'playing') {
            dealerCardsEl.innerHTML = `
                <div class="card">${this.dealerCards[0].rank}${this.dealerCards[0].suit}</div>
                <div class="card card-back">?</div>
            `;
            dealerScoreEl.textContent = this.dealerCards[0].value;
        } else {
            dealerCardsEl.innerHTML = this.dealerCards.map(card => 
                `<div class="card">${card.rank}${card.suit}</div>`
            ).join('');
            dealerScoreEl.textContent = this.calculateScore(this.dealerCards);
        }
        
        playerScoreEl.textContent = this.calculateScore(this.playerCards);
    }
    
    // End blackjack game
    endBlackjackGame(result, betAmount) {
        this.gameState = 'finished';
        
        let winAmount = 0;
        let message = '';
        
        switch (result) {
            case 'blackjack':
                winAmount = betAmount * 2.5;
                message = 'بلک جک! تبریک!';
                break;
            case 'win':
            case 'dealer_bust':
                winAmount = betAmount * 2;
                message = 'شما برنده شدید!';
                break;
            case 'push':
                winAmount = betAmount;
                message = 'مساوی!';
                break;
            case 'lose':
            case 'bust':
                winAmount = 0;
                message = 'شما باختید!';
                break;
        }
        
        if (winAmount > 0) {
            this.balance += winAmount;
            this.updateBalanceDisplay();
        }
        
        document.querySelector('.game-status').textContent = message;
        document.querySelector('.play-controls').style.display = 'none';
        document.querySelector('.bet-controls').style.display = 'block';
        
        this.updateBlackjackDisplay();
        this.recordGameResult('blackjack', betAmount, winAmount - betAmount);
    }
    
    // Roulette game
    getRouletteHTML() {
        return `
            <div class="game-container roulette-game">
                <div class="game-header">
                    <h2>رولت</h2>
                    <button class="close-game">&times;</button>
                </div>
                <div class="game-content">
                    <div class="roulette-wheel">
                        <div class="wheel" id="roulette-wheel">
                            <div class="ball"></div>
                        </div>
                        <div class="winning-number">
                            <span id="winning-number">-</span>
                        </div>
                    </div>
                    <div class="betting-table">
                        <div class="color-bets">
                            <button class="bet-btn bet-red-btn" data-payout="2">قرمز (1:1)</button>
                            <button class="bet-btn bet-black-btn" data-payout="2">سیاه (1:1)</button>
                        </div>
                        <div class="number-grid">
                            ${Array.from({length: 37}, (_, i) => 
                                `<button class="bet-btn bet-number-btn" data-number="${i}" data-payout="36">${i}</button>`
                            ).join('')}
                        </div>
                    </div>
                    <div class="game-controls">
                        <div class="bet-controls">
                            <label>مبلغ شرط:</label>
                            <input type="number" class="bet-amount" min="0.5" max="1000" value="5" step="0.5">
                            <button class="btn btn-primary spin-roulette-btn">چرخاندن</button>
                        </div>
                        <div class="active-bets"></div>
                    </div>
                    <div class="game-info">
                        <div class="balance">موجودی: $<span class="balance-amount">${this.balance}</span></div>
                        <div class="total-bet">کل شرط: $<span class="total-bet-amount">0</span></div>
                    </div>
                </div>
            </div>
        `;
    }
    
    // Close game
    closeGame() {
        const modal = document.querySelector('.game-modal');
        if (modal) {
            modal.classList.remove('show');
            setTimeout(() => {
                document.body.removeChild(modal);
            }, 300);
        }
        
        this.activeGame = null;
        this.isPlaying = false;
    }
    
    // Record game result
    recordGameResult(gameType, betAmount, winAmount) {
        const result = {
            game: gameType,
            bet: betAmount,
            win: winAmount,
            profit: winAmount - betAmount,
            timestamp: Date.now()
        };
        
        this.gameHistory.push(result);
        
        // Keep only last 100 results
        if (this.gameHistory.length > 100) {
            this.gameHistory.shift();
        }
        
        // Send to server (optional)
        this.sendGameResult(result);
    }
    
    // Send game result to server
    async sendGameResult(result) {
        try {
            await fetch('/api/bets.php?endpoint=place', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    game_type: 'casino',
                    bet_amount: result.bet,
                    bet_data: {
                        game_id: result.game,
                        result: result
                    }
                })
            });
        } catch (error) {
            console.error('Error sending game result:', error);
        }
    }
    
    // Update balance display
    updateBalanceDisplay() {
        const balanceElements = document.querySelectorAll('.balance-amount');
        balanceElements.forEach(element => {
            element.textContent = this.balance.toFixed(2);
        });
    }
    
    // Show notification
    showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification`;
        notification.textContent = message;
        
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 5000);
    }
}

// Initialize casino manager
document.addEventListener('DOMContentLoaded', () => {
    window.casinoManager = new CasinoManager();
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CasinoManager;
}

/* --- assets/js/main.js --- */
// assets/js/main.js - Main JavaScript File
// فایل جاوا اسکریپت اصلی برای سایت Crashify

'use strict';

// Global Configuration
window.Crashify = {
    config: {
        apiUrl: '/api',
        wsUrl: 'wss://ws.crashify.com',
        version: '1.0.0',
        debug: false,
        language: 'en',
        theme: 'light',
        currency: 'USD',
        timezone: 'UTC'
    },
    
    // State Management
    state: {
        user: null,
        isLoggedIn: false,
        notifications: [],
        betSlip: [],
        liveUpdates: true,
        websocket: null,
        modals: new Map(),
        dropdowns: new Map()
    },
    
    // Event Emitter
    events: new EventTarget(),
    
    // Utility Functions
    utils: {},
    
    // API Functions
    api: {},
    
    // UI Components
    ui: {},
    
    // Modules
    modules: {}
};

// Utility Functions
Crashify.utils = {
    // DOM Utilities
    $(selector, context = document) {
        return context.querySelector(selector);
    },
    
    $$(selector, context = document) {
        return Array.from(context.querySelectorAll(selector));
    },
    
    createElement(tag, attributes = {}, children = []) {
        const element = document.createElement(tag);
        
        Object.entries(attributes).forEach(([key, value]) => {
            if (key === 'className') {
                element.className = value;
            } else if (key === 'innerHTML') {
                element.innerHTML = value;
            } else if (key.startsWith('data-')) {
                element.setAttribute(key, value);
            } else {
                element[key] = value;
            }
        });
        
        children.forEach(child => {
            if (typeof child === 'string') {
                element.appendChild(document.createTextNode(child));
            } else if (child instanceof Node) {
                element.appendChild(child);
            }
        });
        
        return element;
    },
    
    // Event Utilities
    on(element, event, handler, options = {}) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            element.addEventListener(event, handler, options);
        }
    },
    
    off(element, event, handler) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            element.removeEventListener(event, handler);
        }
    },
    
    trigger(element, event, data = {}) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            const customEvent = new CustomEvent(event, { detail: data });
            element.dispatchEvent(customEvent);
        }
    },
    
    // Animation Utilities
    fadeIn(element, duration = 300) {
        element.style.opacity = '0';
        element.style.display = 'block';
        
        const start = performance.now();
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.opacity = progress;
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    fadeOut(element, duration = 300) {
        const start = performance.now();
        const startOpacity = parseFloat(getComputedStyle(element).opacity);
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.opacity = startOpacity * (1 - progress);
            
            if (progress >= 1) {
                element.style.display = 'none';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    slideDown(element, duration = 300) {
        element.style.height = '0';
        element.style.overflow = 'hidden';
        element.style.display = 'block';
        
        const targetHeight = element.scrollHeight;
        const start = performance.now();
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.height = (targetHeight * progress) + 'px';
            
            if (progress >= 1) {
                element.style.height = '';
                element.style.overflow = '';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    slideUp(element, duration = 300) {
        const startHeight = element.offsetHeight;
        const start = performance.now();
        
        element.style.overflow = 'hidden';
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.height = (startHeight * (1 - progress)) + 'px';
            
            if (progress >= 1) {
                element.style.display = 'none';
                element.style.height = '';
                element.style.overflow = '';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    // Data Utilities
    formatCurrency(amount, currency = 'USD', decimals = 2) {
        const formatter = new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        });
        return formatter.format(amount);
    },
    
    formatNumber(number, decimals = 0) {
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(number);
    },
    
    formatDate(date, options = {}) {
        const defaultOptions = {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        };
        
        return new Intl.DateTimeFormat('en-US', { ...defaultOptions, ...options }).format(new Date(date));
    },
    
    formatTime(seconds) {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        
        if (hours > 0) {
            return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        }
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    },
    
    // Validation Utilities
    validateEmail(email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return regex.test(email);
    },
    
    validatePassword(password) {
        // At least 8 characters, 1 uppercase, 1 lowercase, 1 number
        const regex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/;
        return regex.test(password);
    },
    
    validateBetAmount(amount, min = 0.01, max = 10000) {
        const num = parseFloat(amount);
        return !isNaN(num) && num >= min && num <= max;
    },
    
    // Storage Utilities
    storage: {
        set(key, value, expiry = null) {
            const item = {
                value: value,
                expiry: expiry ? Date.now() + expiry : null
            };
            localStorage.setItem(key, JSON.stringify(item));
        },
        
        get(key) {
            const item = localStorage.getItem(key);
            if (!item) return null;
            
            try {
                const parsed = JSON.parse(item);
                if (parsed.expiry && Date.now() > parsed.expiry) {
                    localStorage.removeItem(key);
                    return null;
                }
                return parsed.value;
            } catch (e) {
                return null;
            }
        },
        
        remove(key) {
            localStorage.removeItem(key);
        },
        
        clear() {
            localStorage.clear();
        }
    },
    
    // URL Utilities
    getUrlParams() {
        return new URLSearchParams(window.location.search);
    },
    
    updateUrl(params, replace = false) {
        const url = new URL(window.location);
        Object.entries(params).forEach(([key, value]) => {
            if (value === null || value === undefined) {
                url.searchParams.delete(key);
            } else {
                url.searchParams.set(key, value);
            }
        });
        
        if (replace) {
            window.history.replaceState({}, '', url);
        } else {
            window.history.pushState({}, '', url);
        }
    },
    
    // Debounce and Throttle
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },
    
    throttle(func, limit) {
        let inThrottle;
        return function executedFunction(...args) {
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    },
    
    // Random Utilities
    generateId() {
        return Math.random().toString(36).substr(2, 9);
    },
    
    randomBetween(min, max) {
        return Math.random() * (max - min) + min;
    },
    
    // Device Detection
    isMobile() {
        return window.innerWidth <= 768;
    },
    
    isTablet() {
        return window.innerWidth > 768 && window.innerWidth <= 1024;
    },
    
    isDesktop() {
        return window.innerWidth > 1024;
    },
    
    // Copy to Clipboard
    async copyToClipboard(text) {
        try {
            await navigator.clipboard.writeText(text);
            return true;
        } catch (err) {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                return true;
            } catch (e) {
                return false;
            } finally {
                document.body.removeChild(textArea);
            }
        }
    }
};

// API Functions
Crashify.api = {
    // Base request function
    async request(endpoint, options = {}) {
        const defaultOptions = {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            credentials: 'same-origin'
        };
        
        const config = { ...defaultOptions, ...options };
        
        // Add CSRF token if available
        const csrfToken = Crashify.utils.$('meta[name="csrf-token"]')?.getAttribute('content');
        if (csrfToken) {
            config.headers['X-CSRF-TOKEN'] = csrfToken;
        }
        
        try {
            const response = await fetch(`${Crashify.config.apiUrl}${endpoint}`, config);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return await response.json();
            }
            
            return await response.text();
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    },
    
    // Authentication
    async login(credentials) {
        return this.request('/auth/login', {
            method: 'POST',
            body: JSON.stringify(credentials)
        });
    },
    
    async register(userData) {
        return this.request('/auth/register', {
            method: 'POST',
            body: JSON.stringify(userData)
        });
    },
    
    async logout() {
        return this.request('/auth/logout', {
            method: 'POST'
        });
    },
    
    // User
    async getProfile() {
        return this.request('/user/profile');
    },
    
    async updateProfile(data) {
        return this.request('/user/profile', {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    },
    
    // Wallet
    async getBalance() {
        return this.request('/wallet/balance');
    },
    
    async deposit(data) {
        return this.request('/wallet/deposit', {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    async withdraw(data) {
        return this.request('/wallet/withdraw', {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    // Betting
    async placeBet(betData) {
        return this.request('/betting/place', {
            method: 'POST',
            body: JSON.stringify(betData)
        });
    },
    
    async getBetHistory(params = {}) {
        const queryString = new URLSearchParams(params).toString();
        return this.request(`/betting/history?${queryString}`);
    },
    
    // Sports
    async getSports() {
        return this.request('/sports');
    },
    
    async getEvents(sportId, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        return this.request(`/sports/${sportId}/events?${queryString}`);
    },
    
    async getOdds(eventId) {
        return this.request(`/events/${eventId}/odds`);
    },
    
    // Notifications
    async getNotifications() {
        return this.request('/notifications');
    },
    
    async markNotificationRead(id) {
        return this.request(`/notifications/${id}/read`, {
            method: 'POST'
        });
    }
};

// UI Components
Crashify.ui = {
    // Toast Notifications
    toast: {
        container: null,
        
        init() {
            if (!this.container) {
                this.container = Crashify.utils.createElement('div', {
                    className: 'toast-container',
                    style: 'position: fixed; top: 20px; right: 20px; z-index: 1080;'
                });
                document.body.appendChild(this.container);
            }
        },
        
        show(message, type = 'info', duration = 5000) {
            this.init();
            
            const toast = Crashify.utils.createElement('div', {
                className: `alert alert-${type} toast-item`,
                style: 'margin-bottom: 10px; min-width: 300px; opacity: 0; transform: translateX(100%); transition: all 0.3s ease;'
            });
            
            const content = Crashify.utils.createElement('div', {
                className: 'alert-content',
                innerHTML: `<i class="fas fa-${this.getIcon(type)}"></i><span>${message}</span>`
            });
            
            const closeBtn = Crashify.utils.createElement('button', {
                className: 'alert-close',
                innerHTML: '<i class="fas fa-times"></i>'
            });
            
            toast.appendChild(content);
            toast.appendChild(closeBtn);
            this.container.appendChild(toast);
            
            // Animate in
            requestAnimationFrame(() => {
                toast.style.opacity = '1';
                toast.style.transform = 'translateX(0)';
            });
            
            // Auto remove
            const timeoutId = setTimeout(() => {
                this.remove(toast);
            }, duration);
            
            // Manual close
            closeBtn.addEventListener('click', () => {
                clearTimeout(timeoutId);
                this.remove(toast);
            });
        },
        
        remove(toast) {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 300);
        },
        
        getIcon(type) {
            const icons = {
                success: 'check-circle',
                error: 'exclamation-circle',
                warning: 'exclamation-triangle',
                info: 'info-circle'
            };
            return icons[type] || 'info-circle';
        },
        
        success(message, duration) {
            this.show(message, 'success', duration);
        },
        
        error(message, duration) {
            this.show(message, 'error', duration);
        },
        
        warning(message, duration) {
            this.show(message, 'warning', duration);
        },
        
        info(message, duration) {
            this.show(message, 'info', duration);
        }
    },
    
    // Loading Overlay
    loading: {
        overlay: null,
        
        show(message = 'Loading...') {
            if (!this.overlay) {
                this.overlay = Crashify.utils.createElement('div', {
                    id: 'loadingOverlay',
                    className: 'loading-overlay',
                    innerHTML: `
                        <div class="loading-spinner">
                            <div class="spinner-ring"></div>
                            <div class="loading-text">${message}</div>
                        </div>
                    `
                });
                document.body.appendChild(this.overlay);
            }
            
            this.overlay.style.display = 'flex';
            Crashify.utils.fadeIn(this.overlay, 200);
        },
        
        hide() {
            if (this.overlay) {
                Crashify.utils.fadeOut(this.overlay, 200);
            }
        },
        
        updateMessage(message) {
            if (this.overlay) {
                const textElement = this.overlay.querySelector('.loading-text');
                if (textElement) {
                    textElement.textContent = message;
                }
            }
        }
    },
    
    // Confirmation Dialog
    confirm(message, title = 'Confirm', options = {}) {
        return new Promise((resolve) => {
            const modal = Crashify.utils.createElement('div', {
                className: 'modal-backdrop active'
            });
            
            const dialog = Crashify.utils.createElement('div', {
                className: 'modal active',
                style: 'max-width: 400px;',
                innerHTML: `
                    <div class="modal-header">
                        <h3 class="modal-title">${title}</h3>
                    </div>
                    <div class="modal-body">
                        <p>${message}</p>
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-outline cancel-btn">${options.cancelText || 'Cancel'}</button>
                        <button class="btn btn-primary confirm-btn">${options.confirmText || 'Confirm'}</button>
                    </div>
                `
            });
            
            modal.appendChild(dialog);
            document.body.appendChild(modal);
            
            const cleanup = () => {
                document.body.removeChild(modal);
            };
            
            dialog.querySelector('.cancel-btn').addEventListener('click', () => {
                cleanup();
                resolve(false);
            });
            
            dialog.querySelector('.confirm-btn').addEventListener('click', () => {
                cleanup();
                resolve(true);
            });
            
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    cleanup();
                    resolve(false);
                }
            });
        });
    }
};

// Theme Management
function initializeTheme(theme = 'light') {
    Crashify.config.theme = theme;
    document.body.className = document.body.className.replace(/theme-\w+/g, '');
    document.body.classList.add(`theme-${theme}`);
    
    // Save theme preference
    Crashify.utils.storage.set('theme', theme);
    
    // Emit theme change event
    Crashify.events.dispatchEvent(new CustomEvent('themeChanged', { detail: { theme } }));
}

function changeTheme(theme) {
    initializeTheme(theme);
    
    // Update theme selector UI
    Crashify.utils.$$('.theme-option').forEach(option => {
        option.classList.remove('active');
        if (option.dataset.theme === theme) {
            option.classList.add('active');
        }
    });
    
    Crashify.ui.toast.success(`Theme changed to ${theme}`);
}

// Language Management
function setLanguage(language, isRTL = false) {
    Crashify.config.language = language;
    document.documentElement.lang = language;
    document.documentElement.dir = isRTL ? 'rtl' : 'ltr';
    document.body.classList.toggle('rtl', isRTL);
    
    // Save language preference
    Crashify.utils.storage.set('language', language);
    
    // Emit language change event
    Crashify.events.dispatchEvent(new CustomEvent('languageChanged', { detail: { language, isRTL } }));
}

function changeLanguage(language) {
    // This would typically reload the page with new language
    window.location.href = `${window.location.pathname}?lang=${language}`;
}

// User Session Management
function initializeUserSession(user) {
    Crashify.state.user = user;
    Crashify.state.isLoggedIn = !!user;
    
    if (user) {
        // Initialize user-specific features
        loadUserNotifications();
        startBalanceUpdates();
    }
}

async function loadUserNotifications() {
    try {
        const notifications = await Crashify.api.getNotifications();
        Crashify.state.notifications = notifications;
        updateNotificationUI();
    } catch (error) {
        console.error('Failed to load notifications:', error);
    }
}

function updateNotificationUI() {
    const badge = Crashify.utils.$('.notification-badge');
    const unreadCount = Crashify.state.notifications.filter(n => !n.read).length;
    
    if (badge) {
        badge.textContent = unreadCount;
        badge.style.display = unreadCount > 0 ? 'block' : 'none';
    }
}

// Modal Management
function openModal(modalId) {
    const modal = Crashify.utils.$(`#${modalId}`);
    if (modal) {
        modal.classList.add('active');
        const backdrop = modal.closest('.modal-backdrop');
        if (backdrop) {
            backdrop.classList.add('active');
        }
        
        // Prevent body scroll
        document.body.style.overflow = 'hidden';
        
        // Focus management
        const firstFocusable = modal.querySelector('input, button, select, textarea, [tabindex]:not([tabindex="-1"])');
        if (firstFocusable) {
            firstFocusable.focus();
        }
        
        Crashify.state.modals.set(modalId, modal);
    }
}

function closeModal(modalId) {
    const modal = Crashify.state.modals.get(modalId) || Crashify.utils.$(`#${modalId}`);
    if (modal) {
        modal.classList.remove('active');
        const backdrop = modal.closest('.modal-backdrop');
        if (backdrop) {
            backdrop.classList.remove('active');
        }
        
        // Restore body scroll
        document.body.style.overflow = '';
        
        Crashify.state.modals.delete(modalId);
    }
}

// Dropdown Management
function toggleDropdown(dropdownId) {
    const dropdown = Crashify.utils.$(`#${dropdownId}`);
    if (dropdown) {
        const isOpen = dropdown.classList.contains('active');
        
        // Close all other dropdowns
        Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
            menu.classList.remove('active');
        });
        
        if (!isOpen) {
            dropdown.classList.add('active');
            Crashify.state.dropdowns.set(dropdownId, dropdown);
        } else {
            Crashify.state.dropdowns.delete(dropdownId);
        }
    }
}

// Mobile Menu
function toggleMobileMenu() {
    const mobileNav = Crashify.utils.$('#mobileNav');
    const menuToggle = Crashify.utils.$('.mobile-menu-toggle');
    
    if (mobileNav && menuToggle) {
        const isOpen = mobileNav.classList.contains('active');
        
        mobileNav.classList.toggle('active');
        menuToggle.classList.toggle('active');
        document.body.classList.toggle('mobile-menu-open');
    }
}

// Scroll to Top
function scrollToTop() {
    window.scrollTo({
        top: 0,
        behavior: 'smooth'
    });
}

// Back to Top Button
function initializeBackToTop() {
    const backToTopBtn = Crashify.utils.$('#backToTop');
    if (backToTopBtn) {
        const toggleVisibility = Crashify.utils.throttle(() => {
            if (window.pageYOffset > 300) {
                backToTopBtn.style.display = 'flex';
            } else {
                backToTopBtn.style.display = 'none';
            }
        }, 100);
        
        window.addEventListener('scroll', toggleVisibility);
        toggleVisibility(); // Initial check
    }
}

// Loading Screen
function hideLoadingScreen() {
    const loadingScreen = Crashify.utils.$('#loadingScreen');
    if (loadingScreen) {
        setTimeout(() => {
            Crashify.utils.fadeOut(loadingScreen, 500);
        }, 500);
    }
}

// Initialize UI Components
function initializeUI() {
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize back to top
    initializeBackToTop();
    
    // Initialize form validation
    initializeFormValidation();
    
    // Initialize keyboard shortcuts
    initializeKeyboardShortcuts();
    
    // Initialize click outside handlers
    initializeClickOutside();
}

function initializeTooltips() {
    // Simple tooltip implementation
    Crashify.utils.$$('[title]').forEach(element => {
        element.addEventListener('mouseenter', showTooltip);
        element.addEventListener('mouseleave', hideTooltip);
    });
}

function showTooltip(event) {
    const element = event.target;
    const title = element.getAttribute('title');
    
    if (title) {
        element.setAttribute('data-original-title', title);
        element.removeAttribute('title');
        
        const tooltip = Crashify.utils.createElement('div', {
            className: 'tooltip',
            innerHTML: title,
            style: 'position: absolute; z-index: 1070; background: #333; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; pointer-events: none;'
        });
        
        document.body.appendChild(tooltip);
        
        const rect = element.getBoundingClientRect();
        tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
        tooltip.style.top = rect.top - tooltip.offsetHeight - 5 + 'px';
        
        element._tooltip = tooltip;
    }
}

function hideTooltip(event) {
    const element = event.target;
    const originalTitle = element.getAttribute('data-original-title');
    
    if (originalTitle) {
        element.setAttribute('title', originalTitle);
        element.removeAttribute('data-original-title');
    }
    
    if (element._tooltip) {
        document.body.removeChild(element._tooltip);
        delete element._tooltip;
    }
}

function initializeFormValidation() {
    Crashify.utils.$$('form[data-validate]').forEach(form => {
        form.addEventListener('submit', validateForm);
    });
}

function validateForm(event) {
    const form = event.target;
    let isValid = true;
    
    // Clear previous errors
    Crashify.utils.$$(form, '.form-error').forEach(error => {
        error.remove();
    });
    
    // Validate required fields
    Crashify.utils.$$(form, '[required]').forEach(field => {
        if (!field.value.trim()) {
            showFieldError(field, 'This field is required');
            isValid = false;
        }
    });
    
    // Validate email fields
    Crashify.utils.$$(form, 'input[type="email"]').forEach(field => {
        if (field.value && !Crashify.utils.validateEmail(field.value)) {
            showFieldError(field, 'Please enter a valid email address');
            isValid = false;
        }
    });
    
    // Validate password fields
    Crashify.utils.$$(form, 'input[data-validate="password"]').forEach(field => {
        if (field.value && !Crashify.utils.validatePassword(field.value)) {
            showFieldError(field, 'Password must be at least 8 characters with uppercase, lowercase, and number');
            isValid = false;
        }
    });
    
    if (!isValid) {
        event.preventDefault();
    }
}

function showFieldError(field, message) {
    const error = Crashify.utils.createElement('div', {
        className: 'form-error',
        innerHTML: message
    });
    
    field.parentNode.appendChild(error);
    field.classList.add('error');
}

function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', (event) => {
        // Escape key closes modals and dropdowns
        if (event.key === 'Escape') {
            // Close modals
            Crashify.state.modals.forEach((modal, id) => {
                closeModal(id);
            });
            
            // Close dropdowns
            Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
                menu.classList.remove('active');
            });
            
            // Close mobile menu
            const mobileNav = Crashify.utils.$('#mobileNav');
            if (mobileNav && mobileNav.classList.contains('active')) {
                toggleMobileMenu();
            }
        }
        
        // Ctrl/Cmd + K for search
        if ((event.ctrlKey || event.metaKey) && event.key === 'k') {
            event.preventDefault();
            const searchInput = Crashify.utils.$('#searchInput');
            if (searchInput) {
                searchInput.focus();
            }
        }
    });
}

function initializeClickOutside() {
    document.addEventListener('click', (event) => {
        // Close dropdowns when clicking outside
        if (!event.target.closest('.dropdown')) {
            Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
                menu.classList.remove('active');
            });
            Crashify.state.dropdowns.clear();
        }
        
        // Close mobile menu when clicking outside
        const mobileNav = Crashify.utils.$('#mobileNav');
        if (mobileNav && mobileNav.classList.contains('active') && !event.target.closest('.mobile-nav, .mobile-menu-toggle')) {
            toggleMobileMenu();
        }
    });
}

// Cookie Consent
function checkCookieConsent() {
    const consent = Crashify.utils.storage.get('cookieConsent');
    if (!consent) {
        const banner = Crashify.utils.$('#cookieBanner');
        if (banner) {
            banner.style.display = 'block';
        }
    }
}

function acceptCookies() {
    Crashify.utils.storage.set('cookieConsent', true);
    const banner = Crashify.utils.$('#cookieBanner');
    if (banner) {
        Crashify.utils.fadeOut(banner);
    }
}

function showCookieSettings() {
    openModal('cookieSettingsModal');
}

// Export global functions
window.initializeTheme = initializeTheme;
window.changeTheme = changeTheme;
window.setLanguage = setLanguage;
window.changeLanguage = changeLanguage;
window.initializeUserSession = initializeUserSession;
window.openModal = openModal;
window.closeModal = closeModal;
window.toggleDropdown = toggleDropdown;
window.toggleMobileMenu = toggleMobileMenu;
window.scrollToTop = scrollToTop;
window.hideLoadingScreen = hideLoadingScreen;
window.initializeUI = initializeUI;
window.checkCookieConsent = checkCookieConsent;
window.acceptCookies = acceptCookies;
window.showCookieSettings = showCookieSettings;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeUI);
} else {
    initializeUI();
}

console.log('Crashify main.js loaded successfully');

/* --- assets/js/wallet.js --- */
// assets/js/wallet.js - Wallet Management JavaScript (Continued)
// جاوااسکریپت مدیریت کیف پول (ادامه)

            `;
        }
        
        container.innerHTML = html;
        container.style.display = html ? 'block' : 'none';
    }
    
    // Show payment modal
    showPaymentModal(paymentData) {
        const modal = document.createElement('div');
        modal.className = 'payment-modal';
        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h5>تکمیل پرداخت</h5>
                    <button class="close-modal">&times;</button>
                </div>
                <div class="modal-body">
                    <div class="payment-info">
                        <div class="payment-amount">
                            <h3>$${paymentData.amount}</h3>
                            <p>مبلغ قابل پرداخت</p>
                        </div>
                        <div class="payment-method">
                            <p><strong>روش پرداخت:</strong> ${paymentData.payment_method}</p>
                        </div>
                        <div class="payment-instructions">
                            ${paymentData.instructions || ''}
                        </div>
                    </div>
                    <div class="payment-actions">
                        <a href="${paymentData.payment_url}" target="_blank" class="btn btn-primary">
                            ادامه پرداخت
                        </a>
                        <button class="btn btn-secondary close-modal">انصراف</button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelectorAll('.close-modal').forEach(btn => {
            btn.addEventListener('click', () => {
                document.body.removeChild(modal);
            });
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Show withdrawal success
    showWithdrawalSuccess(data) {
        const modal = document.createElement('div');
        modal.className = 'success-modal';
        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h5><i class="fas fa-check-circle text-success"></i> درخواست برداشت ثبت شد</h5>
                </div>
                <div class="modal-body">
                    <div class="success-info">
                        <p><strong>مبلغ:</strong> $${data.amount}</p>
                        <p><strong>شماره تراکنش:</strong> ${data.transaction_id}</p>
                        <p><strong>زمان پردازش:</strong> ${data.processing_time}</p>
                        <div class="alert alert-info mt-3">
                            <i class="fas fa-info-circle"></i>
                            درخواست شما در حال بررسی است و پس از تأیید پردازش خواهد شد.
                        </div>
                    </div>
                    <button class="btn btn-primary close-modal">متوجه شدم</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Add close functionality
        modal.querySelector('.close-modal').addEventListener('click', () => {
            document.body.removeChild(modal);
        });
        
        // Show modal
        setTimeout(() => modal.classList.add('show'), 10);
    }
    
    // Render payment methods
    renderPaymentMethods() {
        // Render deposit methods
        const depositSelect = document.querySelector('.deposit-payment-methods');
        if (depositSelect && this.paymentMethods.deposit) {
            let html = '<option value="">انتخاب روش پرداخت</option>';
            this.paymentMethods.deposit.forEach(method => {
                html += `<option value="${method.id}">${method.name}</option>`;
            });
            depositSelect.innerHTML = html;
        }
        
        // Render withdrawal methods
        const withdrawalSelect = document.querySelector('.withdrawal-payment-methods');
        if (withdrawalSelect && this.paymentMethods.withdrawal) {
            let html = '<option value="">انتخاب روش برداشت</option>';
            this.paymentMethods.withdrawal.forEach(method => {
                html += `<option value="${method.id}">${method.name}</option>`;
            });
            withdrawalSelect.innerHTML = html;
        }
    }
    
    // Update limits display
    updateLimitsDisplay() {
        // Update deposit limits
        if (this.limits.deposit) {
            const depositLimits = document.querySelector('.deposit-limits');
            if (depositLimits) {
                depositLimits.innerHTML = `
                    <div class="limit-item">
                        <span>حداقل واریز:</span>
                        <span>$${this.limits.deposit.min_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حداکثر واریز:</span>
                        <span>$${this.limits.deposit.max_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حد روزانه باقی‌مانده:</span>
                        <span>$${this.limits.deposit.daily_remaining}</span>
                    </div>
                `;
            }
        }
        
        // Update withdrawal limits
        if (this.limits.withdrawal) {
            const withdrawalLimits = document.querySelector('.withdrawal-limits');
            if (withdrawalLimits) {
                withdrawalLimits.innerHTML = `
                    <div class="limit-item">
                        <span>حداقل برداشت:</span>
                        <span>$${this.limits.withdrawal.min_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حداکثر برداشت:</span>
                        <span>$${this.limits.withdrawal.max_amount}</span>
                    </div>
                    <div class="limit-item">
                        <span>حد روزانه باقی‌مانده:</span>
                        <span>$${this.limits.withdrawal.daily_remaining}</span>
                    </div>
                `;
            }
        }
    }
    
    // Display recent transactions
    displayRecentTransactions(transactions) {
        const container = document.querySelector('.recent-transactions');
        if (!container) return;
        
        if (transactions.length === 0) {
            container.innerHTML = '<p class="text-center text-muted">تراکنش اخیری یافت نشد</p>';
            return;
        }
        
        let html = '';
        transactions.forEach(transaction => {
            html += this.generateTransactionHTML(transaction);
        });
        
        container.innerHTML = html;
    }
    
    // Update transactions pagination
    updateTransactionsPagination(paginationData) {
        const container = document.querySelector('.transactions-pagination');
        if (!container) return;
        
        let html = '<nav><ul class="pagination justify-content-center">';
        
        // Previous button
        if (paginationData.current_page > 1) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page - 1}">قبلی</a>
            </li>`;
        }
        
        // Page numbers
        const startPage = Math.max(1, paginationData.current_page - 2);
        const endPage = Math.min(paginationData.total_pages, paginationData.current_page + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const isActive = i === paginationData.current_page;
            html += `<li class="page-item ${isActive ? 'active' : ''}">
                <a class="page-link" href="#" data-page="${i}">${i}</a>
            </li>`;
        }
        
        // Next button
        if (paginationData.current_page < paginationData.total_pages) {
            html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${paginationData.current_page + 1}">بعدی</a>
            </li>`;
        }
        
        html += '</ul></nav>';
        container.innerHTML = html;
        
        // Add click handlers
        container.querySelectorAll('.page-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = parseInt(e.target.dataset.page);
                this.loadTransactions(page);
            });
        });
    }
    
    // Start periodic updates
    startPeriodicUpdates() {
        // Update balance every 30 seconds
        setInterval(() => {
            this.loadWalletData();
        }, 30000);
        
        // Check for pending transactions every 60 seconds
        setInterval(() => {
            this.checkPendingTransactions();
        }, 60000);
    }
    
    // Check pending transactions
    async checkPendingTransactions() {
        try {
            const response = await fetch('/api/wallet.php?endpoint=pending-status');
            const data = await response.json();
            
            if (data.success && data.data.updates.length > 0) {
                // Show notifications for updated transactions
                data.data.updates.forEach(update => {
                    if (update.status === 'completed') {
                        this.showNotification(`تراکنش ${update.type} با موفقیت تکمیل شد`, 'success');
                    } else if (update.status === 'failed') {
                        this.showNotification(`تراکنش ${update.type} ناموفق بود`, 'error');
                    }
                });
                
                // Reload wallet data
                this.loadWalletData();
            }
        } catch (error) {
            console.error('Error checking pending transactions:', error);
        }
    }
    
    // Exchange currency
    async exchangeCurrency(fromCurrency, toCurrency, amount) {
        try {
            this.showLoading('در حال تبدیل ارز...');
            
            const response = await fetch('/api/wallet.php?endpoint=exchange', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    from_currency: fromCurrency,
                    to_currency: toCurrency,
                    amount: amount
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification(`تبدیل ارز با موفقیت انجام شد`, 'success');
                this.loadWalletData();
                return result.data;
            } else {
                throw new Error(result.error || 'خطا در تبدیل ارز');
            }
        } catch (error) {
            console.error('Error exchanging currency:', error);
            this.showNotification(error.message, 'error');
            throw error;
        } finally {
            this.hideLoading();
        }
    }
    
    // Get exchange rate
    async getExchangeRate(fromCurrency, toCurrency) {
        try {
            const response = await fetch(`/api/wallet.php?endpoint=exchange-rate&from=${fromCurrency}&to=${toCurrency}`);
            const data = await response.json();
            
            if (data.success) {
                return data.data.rate;
            } else {
                throw new Error(data.error || 'خطا در دریافت نرخ ارز');
            }
        } catch (error) {
            console.error('Error getting exchange rate:', error);
            throw error;
        }
    }
    
    // Generate transaction export
    async exportTransactions(format = 'csv', filters = {}) {
        try {
            this.showLoading('در حال تولید فایل...');
            
            const params = new URLSearchParams({
                format: format,
                ...filters
            });
            
            const response = await fetch(`/api/wallet.php?endpoint=export&${params}`);
            
            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `transactions_${Date.now()}.${format}`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                window.URL.revokeObjectURL(url);
                
                this.showNotification('فایل با موفقیت دانلود شد', 'success');
            } else {
                throw new Error('خطا در تولید فایل');
            }
        } catch (error) {
            console.error('Error exporting transactions:', error);
            this.showNotification(error.message, 'error');
        } finally {
            this.hideLoading();
        }
    }
    
    // Utility functions
    formatCurrency(amount, currency = 'USD') {
        return new Intl.NumberFormat('fa-IR', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: 2
        }).format(amount);
    }
    
    formatDate(dateString) {
        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        }).format(new Date(dateString));
    }
    
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} notification fade-in`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">&times;</button>
        `;
        
        // Add to container
        let container = document.querySelector('.notifications-container');
        if (!container) {
            container = document.createElement('div');
            container.className = 'notifications-container';
            document.body.appendChild(container);
        }
        
        container.appendChild(notification);
        
        // Add close functionality
        notification.querySelector('.notification-close').addEventListener('click', () => {
            this.removeNotification(notification);
        });
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            this.removeNotification(notification);
        }, 5000);
    }
    
    removeNotification(notification) {
        if (notification.parentNode) {
            notification.classList.add('fade-out');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }
    }
    
    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'exclamation-circle',
            'warning': 'exclamation-triangle',
            'info': 'info-circle'
        };
        
        return icons[type] || 'info-circle';
    }
    
    showLoading(message = 'در حال بارگذاری...') {
        let loader = document.querySelector('.wallet-loading-overlay');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'wallet-loading-overlay';
            loader.innerHTML = `
                <div class="loading-content">
                    <div class="loading-spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(loader);
        } else {
            loader.querySelector('.loading-message').textContent = message;
        }
        
        loader.style.display = 'flex';
        this.isLoading = true;
    }
    
    hideLoading() {
        const loader = document.querySelector('.wallet-loading-overlay');
        if (loader) {
            loader.style.display = 'none';
        }
        this.isLoading = false;
    }
    
    // Event system for wallet updates
    addEventListener(event, callback) {
        if (!this.eventListeners) {
            this.eventListeners = new Map();
        }
        
        if (!this.eventListeners.has(event)) {
            this.eventListeners.set(event, []);
        }
        
        this.eventListeners.get(event).push(callback);
    }
    
    removeEventListener(event, callback) {
        if (this.eventListeners && this.eventListeners.has(event)) {
            const listeners = this.eventListeners.get(event);
            const index = listeners.indexOf(callback);
            if (index > -1) {
                listeners.splice(index, 1);
            }
        }
    }
    
    triggerEvent(event, data) {
        if (this.eventListeners && this.eventListeners.has(event)) {
            this.eventListeners.get(event).forEach(callback => {
                try {
                    callback(data);
                } catch (error) {
                    console.error('Error in wallet event listener:', error);
                }
            });
        }
    }
    
    // Cleanup
    destroy() {
        // Clear intervals
        if (this.updateInterval) {
            clearInterval(this.updateInterval);
        }
        
        if (this.pendingCheckInterval) {
            clearInterval(this.pendingCheckInterval);
        }
        
        // Clear event listeners
        if (this.eventListeners) {
            this.eventListeners.clear();
        }
        
        // Remove loading overlay
        const loader = document.querySelector('.wallet-loading-overlay');
        if (loader) {
            document.body.removeChild(loader);
        }
        
        // Remove notifications container
        const notifications = document.querySelector('.notifications-container');
        if (notifications) {
            document.body.removeChild(notifications);
        }
    }
}

// Initialize wallet manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.walletManager = new WalletManager();
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    if (window.walletManager) {
        window.walletManager.destroy();
    }
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = WalletManager;
}

