<?php
// pages/bet-history.php - User Bet History Page
// صفحه تاریخچه شرط‌بندی کاربر

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Require authentication
requireAuth();

// Page settings
$pageTitle = 'Bet History - Crashify';
$bodyClass = 'bet-history-page';
$breadcrumbs = [
    ['title' => 'Account', 'url' => '/pages/dashboard.php'],
    ['title' => 'Bet History']
];

// Get filter parameters
$gameType = $_GET['game'] ?? 'all';
$status = $_GET['status'] ?? 'all';
$dateRange = $_GET['date'] ?? 'all';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 20;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = ["user_id = ?"];
$params = [$currentUser['id']];

if ($gameType !== 'all') {
    $whereConditions[] = "game_type = ?";
    $params[] = $gameType;
}

if ($status !== 'all') {
    $whereConditions[] = "status = ?";
    $params[] = $status;
}

// Date range filter
switch ($dateRange) {
    case 'today':
        $whereConditions[] = "DATE(created_at) = CURDATE()";
        break;
    case 'week':
        $whereConditions[] = "created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        break;
    case 'month':
        $whereConditions[] = "created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        break;
    case 'year':
        $whereConditions[] = "created_at >= DATE_SUB(NOW(), INTERVAL 365 DAY)";
        break;
}

$whereClause = implode(' AND ', $whereConditions);

// Get total count for pagination
$totalBets = fetchValue(
    "SELECT COUNT(*) FROM user_bets WHERE $whereClause",
    $params
);

// Get bets with pagination
$bets = fetchAll(
    "SELECT * FROM user_bets 
     WHERE $whereClause 
     ORDER BY created_at DESC 
     LIMIT $limit OFFSET $offset",
    $params
);

// Calculate pagination
$totalPages = ceil($totalBets / $limit);

// Get summary statistics
$stats = fetchRow(
    "SELECT 
        COUNT(*) as total_bets,
        SUM(bet_amount) as total_wagered,
        SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END) as total_winnings,
        SUM(CASE WHEN status = 'won' THEN win_amount - bet_amount ELSE -bet_amount END) as net_profit,
        COUNT(CASE WHEN status = 'won' THEN 1 END) as wins,
        COUNT(CASE WHEN status = 'lost' THEN 1 END) as losses,
        MAX(win_amount) as biggest_win,
        MAX(CASE WHEN status = 'won' THEN win_amount - bet_amount ELSE 0 END) as biggest_profit
     FROM user_bets 
     WHERE $whereClause",
    $params
) ?: [
    'total_bets' => 0,
    'total_wagered' => 0,
    'total_winnings' => 0,
    'net_profit' => 0,
    'wins' => 0,
    'losses' => 0,
    'biggest_win' => 0,
    'biggest_profit' => 0
];

// Game type options
$gameTypes = [
    'all' => 'All Games',
    'crash' => 'Crash Game',
    'sports' => 'Sports Betting',
    'casino' => 'Casino Games',
    'live' => 'Live Betting'
];

// Status options
$statusOptions = [
    'all' => 'All Status',
    'pending' => 'Pending',
    'won' => 'Won',
    'lost' => 'Lost',
    'cancelled' => 'Cancelled',
    'cashed_out' => 'Cashed Out'
];

// Date range options
$dateRangeOptions = [
    'all' => 'All Time',
    'today' => 'Today',
    'week' => 'This Week',
    'month' => 'This Month',
    'year' => 'This Year'
];

include '../includes/header.php';
?>

<!-- Bet History Header -->
<section class="page-header">
    <div class="container">
        <div class="header-content">
            <div class="page-info">
                <h1 class="page-title">
                    <i class="fas fa-history"></i>
                    Bet History
                </h1>
                <p class="page-description">
                    View and analyze your complete betting history and statistics.
                </p>
            </div>
            
            <div class="header-actions">
                <button class="btn btn-outline" onclick="exportBetHistory()">
                    <i class="fas fa-download"></i>
                    Export CSV
                </button>
                <button class="btn btn-primary" onclick="refreshHistory()">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
</section>

<!-- Statistics Summary -->
<section class="bet-statistics">
    <div class="container">
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-chart-bar"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value"><?php echo number_format($stats['total_bets']); ?></div>
                    <div class="stat-label">Total Bets</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-coins"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">$<?php echo number_format($stats['total_wagered'], 2); ?></div>
                    <div class="stat-label">Total Wagered</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-trophy"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">$<?php echo number_format($stats['total_winnings'], 2); ?></div>
                    <div class="stat-label">Total Winnings</div>
                </div>
            </div>
            
            <div class="stat-card <?php echo $stats['net_profit'] >= 0 ? 'positive' : 'negative'; ?>">
                <div class="stat-icon">
                    <i class="fas fa-<?php echo $stats['net_profit'] >= 0 ? 'arrow-up' : 'arrow-down'; ?>"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">
                        <?php echo $stats['net_profit'] >= 0 ? '+' : ''; ?>$<?php echo number_format($stats['net_profit'], 2); ?>
                    </div>
                    <div class="stat-label">Net Profit</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-percentage"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">
                        <?php echo $stats['total_bets'] > 0 ? round(($stats['wins'] / $stats['total_bets']) * 100, 1) : 0; ?>%
                    </div>
                    <div class="stat-label">Win Rate</div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-star"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-value">$<?php echo number_format($stats['biggest_win'], 2); ?></div>
                    <div class="stat-label">Biggest Win</div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Filters -->
<section class="bet-filters">
    <div class="container">
        <form class="filters-form" method="GET" id="filtersForm">
            <div class="filter-group">
                <label for="gameFilter">Game Type</label>
                <select id="gameFilter" name="game" onchange="applyFilters()">
                    <?php foreach ($gameTypes as $value => $label): ?>
                    <option value="<?php echo $value; ?>" <?php echo $gameType === $value ? 'selected' : ''; ?>>
                        <?php echo $label; ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-group">
                <label for="statusFilter">Status</label>
                <select id="statusFilter" name="status" onchange="applyFilters()">
                    <?php foreach ($statusOptions as $value => $label): ?>
                    <option value="<?php echo $value; ?>" <?php echo $status === $value ? 'selected' : ''; ?>>
                        <?php echo $label; ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-group">
                <label for="dateFilter">Date Range</label>
                <select id="dateFilter" name="date" onchange="applyFilters()">
                    <?php foreach ($dateRangeOptions as $value => $label): ?>
                    <option value="<?php echo $value; ?>" <?php echo $dateRange === $value ? 'selected' : ''; ?>>
                        <?php echo $label; ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="filter-group">
                <button type="button" class="btn btn-outline" onclick="clearFilters()">
                    <i class="fas fa-times"></i>
                    Clear
                </button>
            </div>
        </form>
        
        <div class="results-info">
            <span class="results-count">
                Showing <?php echo count($bets); ?> of <?php echo number_format($totalBets); ?> bets
            </span>
        </div>
    </div>
</section>

<!-- Bet History Table -->
<section class="bet-history-content">
    <div class="container">
        <?php if (empty($bets)): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-history"></i>
            </div>
            <h3>No Bets Found</h3>
            <p>You haven't placed any bets matching the current filters.</p>
            <div class="empty-actions">
                <a href="/pages/casino.php" class="btn btn-primary">
                    <i class="fas fa-dice"></i>
                    Start Playing
                </a>
                <button class="btn btn-outline" onclick="clearFilters()">
                    <i class="fas fa-filter"></i>
                    Clear Filters
                </button>
            </div>
        </div>
        <?php else: ?>
        <div class="bets-table-container">
            <table class="bets-table">
                <thead>
                    <tr>
                        <th>Date & Time</th>
                        <th>Game</th>
                        <th>Bet Details</th>
                        <th>Amount</th>
                        <th>Multiplier/Odds</th>
                        <th>Payout</th>
                        <th>Profit/Loss</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($bets as $bet): ?>
                    <tr class="bet-row" data-bet-id="<?php echo $bet['id']; ?>">
                        <td>
                            <div class="bet-date">
                                <div class="date"><?php echo date('M j, Y', strtotime($bet['created_at'])); ?></div>
                                <div class="time"><?php echo date('H:i:s', strtotime($bet['created_at'])); ?></div>
                            </div>
                        </td>
                        
                        <td>
                            <div class="game-info">
                                <div class="game-type">
                                    <i class="fas fa-<?php echo getGameIcon($bet['game_type']); ?>"></i>
                                    <?php echo ucfirst($bet['game_type']); ?>
                                </div>
                                <?php if ($bet['game_details']): ?>
                                <div class="game-details"><?php echo htmlspecialchars($bet['game_details']); ?></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        
                        <td>
                            <div class="bet-details">
                                <?php if ($bet['selection']): ?>
                                <div class="selection"><?php echo htmlspecialchars($bet['selection']); ?></div>
                                <?php endif; ?>
                                <?php if ($bet['market']): ?>
                                <div class="market"><?php echo htmlspecialchars($bet['market']); ?></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        
                        <td>
                            <div class="bet-amount">
                                $<?php echo number_format($bet['bet_amount'], 2); ?>
                            </div>
                        </td>
                        
                        <td>
                            <div class="odds-multiplier">
                                <?php if ($bet['multiplier']): ?>
                                <?php echo number_format($bet['multiplier'], 2); ?>x
                                <?php elseif ($bet['odds']): ?>
                                <?php echo number_format($bet['odds'], 2); ?>
                                <?php else: ?>
                                -
                                <?php endif; ?>
                            </div>
                        </td>
                        
                        <td>
                            <div class="payout-amount">
                                <?php if ($bet['win_amount'] > 0): ?>
                                $<?php echo number_format($bet['win_amount'], 2); ?>
                                <?php else: ?>
                                $0.00
                                <?php endif; ?>
                            </div>
                        </td>
                        
                        <td>
                            <div class="profit-loss <?php echo getProfitClass($bet); ?>">
                                <?php
                                $profit = 0;
                                if ($bet['status'] === 'won') {
                                    $profit = $bet['win_amount'] - $bet['bet_amount'];
                                } elseif ($bet['status'] === 'lost') {
                                    $profit = -$bet['bet_amount'];
                                } elseif ($bet['status'] === 'cashed_out') {
                                    $profit = $bet['win_amount'] - $bet['bet_amount'];
                                }
                                
                                echo ($profit >= 0 ? '+' : '') . '$' . number_format($profit, 2);
                                ?>
                            </div>
                        </td>
                        
                        <td>
                            <span class="status-badge status-<?php echo $bet['status']; ?>">
                                <?php echo getStatusLabel($bet['status']); ?>
                            </span>
                        </td>
                        
                        <td>
                            <div class="bet-actions">
                                <button class="btn btn-outline btn-xs" onclick="viewBetDetails(<?php echo $bet['id']; ?>)">
                                    <i class="fas fa-eye"></i>
                                    View
                                </button>
                                <?php if ($bet['status'] === 'pending'): ?>
                                <button class="btn btn-danger btn-xs" onclick="cancelBet(<?php echo $bet['id']; ?>)">
                                    <i class="fas fa-times"></i>
                                    Cancel
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination-container">
            <nav class="pagination">
                <?php if ($page > 1): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>" class="page-link prev">
                    <i class="fas fa-chevron-left"></i>
                    Previous
                </a>
                <?php endif; ?>
                
                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                
                if ($startPage > 1): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => 1])); ?>" class="page-link">1</a>
                <?php if ($startPage > 2): ?>
                <span class="page-dots">...</span>
                <?php endif; ?>
                <?php endif; ?>
                
                <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>" 
                   class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                    <?php echo $i; ?>
                </a>
                <?php endfor; ?>
                
                <?php if ($endPage < $totalPages): ?>
                <?php if ($endPage < $totalPages - 1): ?>
                <span class="page-dots">...</span>
                <?php endif; ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $totalPages])); ?>" class="page-link"><?php echo $totalPages; ?></a>
                <?php endif; ?>
                
                <?php if ($page < $totalPages): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>" class="page-link next">
                    Next
                    <i class="fas fa-chevron-right"></i>
                </a>
                <?php endif; ?>
            </nav>
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<!-- Bet Details Modal -->
<div class="modal-backdrop" id="betDetailsModal">
    <div class="modal bet-details-modal">
        <div class="modal-header">
            <h3 class="modal-title">Bet Details</h3>
            <button class="modal-close" onclick="closeModal('betDetailsModal')">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="betDetailsContent">
            <!-- Bet details will be loaded here -->
        </div>
    </div>
</div>

<script>
function applyFilters() {
    document.getElementById('filtersForm').submit();
}

function clearFilters() {
    window.location.href = window.location.pathname;
}

function refreshHistory() {
    window.location.reload();
}

function exportBetHistory() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', 'csv');
    
    window.open(`/api/bets/export?${params.toString()}`, '_blank');
}

function viewBetDetails(betId) {
    Crashify.ui.showLoading();
    
    Crashify.api.request(`/api/bets/${betId}`)
        .then(response => {
            if (response.success) {
                showBetDetailsModal(response.bet);
            } else {
                throw new Error(response.message);
            }
        })
        .catch(error => {
            Crashify.ui.toast.error(error.message || 'Failed to load bet details');
        })
        .finally(() => {
            Crashify.ui.hideLoading();
        });
}

function showBetDetailsModal(bet) {
    const content = `
        <div class="bet-details-content">
            <div class="detail-section">
                <h4>Basic Information</h4>
                <div class="detail-grid">
                    <div class="detail-item">
                        <label>Bet ID:</label>
                        <span>#${bet.id}</span>
                    </div>
                    <div class="detail-item">
                        <label>Date:</label>
                        <span>${new Date(bet.created_at).toLocaleString()}</span>
                    </div>
                    <div class="detail-item">
                        <label>Game Type:</label>
                        <span>${bet.game_type}</span>
                    </div>
                    <div class="detail-item">
                        <label>Status:</label>
                        <span class="status-badge status-${bet.status}">${bet.status}</span>
                    </div>
                </div>
            </div>
            
            <div class="detail-section">
                <h4>Bet Information</h4>
                <div class="detail-grid">
                    <div class="detail-item">
                        <label>Bet Amount:</label>
                        <span>$${parseFloat(bet.bet_amount).toFixed(2)}</span>
                    </div>
                    <div class="detail-item">
                        <label>Odds/Multiplier:</label>
                        <span>${bet.multiplier ? bet.multiplier + 'x' : (bet.odds || '-')}</span>
                    </div>
                    <div class="detail-item">
                        <label>Potential Win:</label>
                        <span>$${(bet.bet_amount * (bet.multiplier || bet.odds || 1)).toFixed(2)}</span>
                    </div>
                    <div class="detail-item">
                        <label>Actual Payout:</label>
                        <span>$${parseFloat(bet.win_amount || 0).toFixed(2)}</span>
                    </div>
                </div>
            </div>
            
            ${bet.selection || bet.market ? `
            <div class="detail-section">
                <h4>Selection Details</h4>
                <div class="detail-grid">
                    ${bet.selection ? `
                    <div class="detail-item">
                        <label>Selection:</label>
                        <span>${bet.selection}</span>
                    </div>
                    ` : ''}
                    ${bet.market ? `
                    <div class="detail-item">
                        <label>Market:</label>
                        <span>${bet.market}</span>
                    </div>
                    ` : ''}
                </div>
            </div>
            ` : ''}
        </div>
    `;
    
    document.getElementById('betDetailsContent').innerHTML = content;
    openModal('betDetailsModal');
}

function cancelBet(betId) {
    if (!confirm('Are you sure you want to cancel this bet?')) {
        return;
    }
    
    Crashify.ui.showLoading();
    
    Crashify.api.request(`/api/bets/${betId}/cancel`, {
        method: 'POST'
    })
    .then(response => {
        if (response.success) {
            Crashify.ui.toast.success('Bet cancelled successfully');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            throw new Error(response.message);
        }
    })
    .catch(error => {
        Crashify.ui.toast.error(error.message || 'Failed to cancel bet');
    })
    .finally(() => {
        Crashify.ui.hideLoading();
    });
}
</script>

<?php
// Helper functions
function getGameIcon($gameType) {
    $icons = [
        'crash' => 'rocket',
        'sports' => 'futbol',
        'casino' => 'dice',
        'live' => 'broadcast-tower'
    ];
    return $icons[$gameType] ?? 'gamepad';
}

function getProfitClass($bet) {
    if ($bet['status'] === 'won' || $bet['status'] === 'cashed_out') {
        return 'profit';
    } elseif ($bet['status'] === 'lost') {
        return 'loss';
    }
    return '';
}

function getStatusLabel($status) {
    $labels = [
        'pending' => 'Pending',
        'won' => 'Won',
        'lost' => 'Lost',
        'cancelled' => 'Cancelled',
        'cashed_out' => 'Cashed Out'
    ];
    return $labels[$status] ?? ucfirst($status);
}

include '../includes/footer.php';
?>
