<?php
// config/languages.php - Language Definitions
// تعریف زبان‌های پشتیبانی شده در سایت Crashify

class Language {
    private $currentLanguage;
    private $translations = [];
    private $fallbackLanguage = 'en';

    public function __construct($language = null) {
        $this->currentLanguage = $language ?: $this->detectLanguage();
        $this->loadTranslations();
    }

    private function detectLanguage() {
        // Check session
        if (isset($_SESSION['language']) && $this->isValidLanguage($_SESSION['language'])) {
            return $_SESSION['language'];
        }

        // Check cookie
        if (isset($_COOKIE['language']) && $this->isValidLanguage($_COOKIE['language'])) {
            return $_COOKIE['language'];
        }

        // Check browser language
        if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $browserLangs = explode(',', $_SERVER['HTTP_ACCEPT_LANGUAGE']);
            foreach ($browserLangs as $lang) {
                $lang = substr(trim($lang), 0, 2);
                if ($this->isValidLanguage($lang)) {
                    return $lang;
                }
            }
        }

        return DEFAULT_LANGUAGE;
    }

    private function isValidLanguage($lang) {
        return array_key_exists($lang, SUPPORTED_LANGUAGES);
    }

    private function loadTranslations() {
        $langFile = __DIR__ . "/../languages/{$this->currentLanguage}.php";
        if (file_exists($langFile)) {
            $this->translations = include $langFile;
        }

        // Load fallback if current language file is incomplete
        if ($this->currentLanguage !== $this->fallbackLanguage) {
            $fallbackFile = __DIR__ . "/../languages/{$this->fallbackLanguage}.php";
            if (file_exists($fallbackFile)) {
                $fallbackTranslations = include $fallbackFile;
                $this->translations = array_merge($fallbackTranslations, $this->translations);
            }
        }
    }

    public function get($key, $params = []) {
        $translation = $this->getNestedValue($this->translations, $key);
        
        if ($translation === null) {
            return $key; // Return key if translation not found
        }

        // Replace parameters
        if (!empty($params)) {
            foreach ($params as $param => $value) {
                $translation = str_replace("{{$param}}", $value, $translation);
            }
        }

        return $translation;
    }

    private function getNestedValue($array, $key) {
        $keys = explode('.', $key);
        $value = $array;

        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return null;
            }
            $value = $value[$k];
        }

        return $value;
    }

    public function setLanguage($language) {
        if ($this->isValidLanguage($language)) {
            $this->currentLanguage = $language;
            $_SESSION['language'] = $language;
            setcookie('language', $language, time() + (365 * 24 * 60 * 60), '/');
            $this->loadTranslations();
            return true;
        }
        return false;
    }

    public function getCurrentLanguage() {
        return $this->currentLanguage;
    }

    public function getSupportedLanguages() {
        return SUPPORTED_LANGUAGES;
    }

    public function isRTL() {
        return in_array($this->currentLanguage, ['ar', 'fa']);
    }

    public function getDirection() {
        return $this->isRTL() ? 'rtl' : 'ltr';
    }

    public function formatNumber($number, $decimals = 2) {
        $locales = [
            'en' => ['decimal' => '.', 'thousands' => ','],
            'fa' => ['decimal' => '/', 'thousands' => '،'],
            'ar' => ['decimal' => '.', 'thousands' => ','],
            'tr' => ['decimal' => ',', 'thousands' => '.'],
            'ru' => ['decimal' => ',', 'thousands' => ' '],
            'de' => ['decimal' => ',', 'thousands' => '.'],
            'fr' => ['decimal' => ',', 'thousands' => ' '],
            'es' => ['decimal' => ',', 'thousands' => '.']
        ];

        $locale = $locales[$this->currentLanguage] ?? $locales['en'];
        return number_format($number, $decimals, $locale['decimal'], $locale['thousands']);
    }

    public function formatCurrency($amount, $currency = 'USD') {
        $symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'BTC' => '₿',
            'ETH' => 'Ξ',
            'USDT' => '₮'
        ];

        $symbol = $symbols[$currency] ?? $currency;
        $formattedAmount = $this->formatNumber($amount);

        // Different currency formats for different languages
        switch ($this->currentLanguage) {
            case 'fa':
                return $formattedAmount . ' ' . $symbol;
            case 'ar':
                return $symbol . ' ' . $formattedAmount;
            default:
                return $symbol . $formattedAmount;
        }
    }

    public function formatDate($timestamp, $format = 'default') {
        $formats = [
            'en' => [
                'default' => 'M j, Y H:i',
                'short' => 'M j',
                'long' => 'F j, Y \a\t H:i'
            ],
            'fa' => [
                'default' => 'j F Y H:i',
                'short' => 'j F',
                'long' => 'j F Y \s\a\a\t H:i'
            ],
            'ar' => [
                'default' => 'j F Y H:i',
                'short' => 'j F',
                'long' => 'j F Y \a\t H:i'
            ]
        ];

        $langFormats = $formats[$this->currentLanguage] ?? $formats['en'];
        $dateFormat = $langFormats[$format] ?? $langFormats['default'];

        return date($dateFormat, $timestamp);
    }
}

// Global language instance
$lang = new Language();

// Helper function for translations
function __($key, $params = []) {
    global $lang;
    return $lang->get($key, $params);
}

function _e($key, $params = []) {
    echo __($key, $params);
}

function getCurrentLanguage() {
    global $lang;
    return $lang->getCurrentLanguage();
}

function isRTL() {
    global $lang;
    return $lang->isRTL();
}

function getDirection() {
    global $lang;
    return $lang->getDirection();
}
?>
