<?php
// pages/live-betting.php - Live Betting Page
// صفحه شرط‌بندی زنده

require_once '../config/config.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Page settings
$pageTitle = 'Live Betting - In-Play Sports Betting';
$bodyClass = 'live-betting-page';
$breadcrumbs = [
    ['title' => 'Sportsbook', 'url' => '/pages/sportsbook.php'],
    ['title' => 'Live Betting']
];

// Get filters from URL
$selectedSport = $_GET['sport'] ?? 'all';
$sortBy = $_GET['sort'] ?? 'time';
$searchQuery = $_GET['search'] ?? '';

// Get live events
$liveEvents = getLiveEvents(100);

// Filter events
if ($selectedSport !== 'all') {
    $liveEvents = array_filter($liveEvents, fn($event) => $event['sport_slug'] === $selectedSport);
}

if ($searchQuery) {
    $liveEvents = array_filter($liveEvents, function($event) use ($searchQuery) {
        return stripos($event['home_team'], $searchQuery) !== false ||
               stripos($event['away_team'], $searchQuery) !== false ||
               stripos($event['league_name'], $searchQuery) !== false;
    });
}

// Sort events
usort($liveEvents, function($a, $b) use ($sortBy) {
    switch ($sortBy) {
        case 'league':
            return strcmp($a['league_name'], $b['league_name']);
        case 'sport':
            return strcmp($a['sport_name'], $b['sport_name']);
        case 'time':
        default:
            return strcmp($a['start_time'], $b['start_time']);
    }
});

// Get sports with live events
$liveSports = [];
foreach ($liveEvents as $event) {
    $sportSlug = $event['sport_slug'] ?? strtolower(str_replace(' ', '-', $event['sport_name']));
    if (!isset($liveSports[$sportSlug])) {
        $liveSports[$sportSlug] = [
            'name' => $event['sport_name'],
            'icon' => $event['sport_icon'],
            'count' => 0
        ];
    }
    $liveSports[$sportSlug]['count']++;
}

$additionalScripts = [
    '/assets/js/betting.js',
    '/assets/js/live-betting.js'
];

include '../includes/header.php';
?>

<!-- Hero Section -->
<section class="live-betting-hero">
    <div class="container">
        <div class="hero-content">
            <div class="hero-text">
                <h1 class="hero-title">
                    <i class="fas fa-broadcast-tower"></i>
                    Live Betting
                    <span class="live-indicator">LIVE</span>
                </h1>
                <p class="hero-description">
                    Experience the thrill of in-play betting with real-time odds updates, 
                    live streaming, and instant bet placement on ongoing matches.
                </p>
                <div class="hero-stats">
                    <div class="stat-item">
                        <div class="stat-number live-counter" data-count="<?php echo count($liveEvents); ?>">
                            <?php echo count($liveEvents); ?>
                        </div>
                        <div class="stat-label">Live Events</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo count($liveSports); ?></div>
                        <div class="stat-label">Sports</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number">24/7</div>
                        <div class="stat-label">Coverage</div>
                    </div>
                </div>
            </div>
            <div class="hero-visual">
                <div class="live-match-card">
                    <div class="match-header">
                        <span class="live-badge pulsing">LIVE</span>
                        <span class="match-time">78'</span>
                    </div>
                    <div class="match-teams">
                        <div class="team">
                            <img src="/assets/images/teams/manchester-united.png" alt="Manchester United">
                            <span class="team-name">Man United</span>
                            <span class="score">2</span>
                        </div>
                        <div class="vs">-</div>
                        <div class="team">
                            <img src="/assets/images/teams/liverpool.png" alt="Liverpool">
                            <span class="team-name">Liverpool</span>
                            <span class="score">1</span>
                        </div>
                    </div>
                    <div class="live-odds">
                        <button class="odds-btn">2.10</button>
                        <button class="odds-btn">3.40</button>
                        <button class="odds-btn">3.20</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Live Sports Navigation -->
<section class="live-sports-nav">
    <div class="container">
        <div class="sports-tabs">
            <a href="?sport=all" class="sport-tab <?php echo $selectedSport === 'all' ? 'active' : ''; ?>">
                <i class="fas fa-globe"></i>
                <span>All Sports</span>
                <span class="event-count"><?php echo count($liveEvents); ?></span>
            </a>
            <?php foreach ($liveSports as $slug => $sport): ?>
            <a href="?sport=<?php echo $slug; ?>" 
               class="sport-tab <?php echo $selectedSport === $slug ? 'active' : ''; ?>">
                <i class="<?php echo $sport['icon']; ?>"></i>
                <span><?php echo $sport['name']; ?></span>
                <span class="event-count"><?php echo $sport['count']; ?></span>
            </a>
            <?php endforeach; ?>
        </div>
        
        <div class="nav-actions">
            <div class="auto-refresh">
                <label class="toggle-switch">
                    <input type="checkbox" id="autoRefresh" checked>
                    <span class="toggle-slider"></span>
                </label>
                <span>Auto Refresh</span>
            </div>
            <button class="refresh-btn" onclick="refreshLiveData()">
                <i class="fas fa-sync-alt"></i>
                <span>Refresh</span>
            </button>
        </div>
    </div>
</section>

<!-- Live Events -->
<section class="live-events-section">
    <div class="container">
        <div class="section-header">
            <h2 class="section-title">
                <i class="fas fa-circle live-dot"></i>
                Live Events
                <?php if ($searchQuery): ?>
                <span class="search-results">- Results for "<?php echo htmlspecialchars($searchQuery); ?>"</span>
                <?php endif; ?>
            </h2>
            <div class="results-count">
                <span class="count"><?php echo count($liveEvents); ?></span>
                <span class="label">live events</span>
                <div class="last-update">
                    Last updated: <span id="lastUpdate"><?php echo date('H:i:s'); ?></span>
                </div>
            </div>
        </div>
        
        <?php if (empty($liveEvents)): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-broadcast-tower"></i>
            </div>
            <h3>No Live Events</h3>
            <p>There are currently no live events. Check back soon!</p>
            <div class="empty-actions">
                <a href="/pages/sportsbook.php" class="btn btn-outline">View Upcoming Events</a>
            </div>
        </div>
        <?php else: ?>
        <div class="live-events-container" id="liveEventsContainer">
            <?php foreach ($liveEvents as $event): ?>
            <div class="live-event-card" data-event-id="<?php echo $event['id']; ?>">
                <div class="event-header">
                    <div class="event-meta">
                        <div class="sport-league">
                            <i class="<?php echo $event['sport_icon']; ?>"></i>
                            <span class="sport-name"><?php echo $event['sport_name']; ?></span>
                            <span class="separator">•</span>
                            <span class="league-name"><?php echo $event['league_name']; ?></span>
                        </div>
                        <div class="live-status">
                            <span class="live-badge pulsing">LIVE</span>
                            <span class="match-time"><?php echo $event['match_time']; ?>'</span>
                        </div>
                    </div>
                    
                    <div class="event-actions">
                        <button class="action-btn favorite" onclick="toggleFavorite(<?php echo $event['id']; ?>)">
                            <i class="far fa-star"></i>
                        </button>
                        <?php if ($event['live_streaming']): ?>
                        <button class="action-btn stream" onclick="openStream(<?php echo $event['id']; ?>)">
                            <i class="fas fa-play"></i>
                            <span>Stream</span>
                        </button>
                        <?php endif; ?>
                        <button class="action-btn stats" onclick="showMatchStats(<?php echo $event['id']; ?>)">
                            <i class="fas fa-chart-bar"></i>
                            <span>Stats</span>
                        </button>
                    </div>
                </div>
                
                <div class="match-info">
                    <div class="teams-section">
                        <div class="team home">
                            <div class="team-info">
                                <img src="<?php echo $event['home_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                                     alt="<?php echo $event['home_team']; ?>" class="team-logo">
                                <span class="team-name"><?php echo $event['home_team']; ?></span>
                            </div>
                            <div class="team-score"><?php echo $event['home_score']; ?></div>
                        </div>
                        
                        <div class="match-center">
                            <div class="score-display">
                                <span class="score"><?php echo $event['home_score']; ?> - <?php echo $event['away_score']; ?></span>
                            </div>
                        </div>
                        
                        <div class="team away">
                            <div class="team-score"><?php echo $event['away_score']; ?></div>
                            <div class="team-info">
                                <img src="<?php echo $event['away_team_logo'] ?: '/assets/images/default-team.png'; ?>" 
                                     alt="<?php echo $event['away_team']; ?>" class="team-logo">
                                <span class="team-name"><?php echo $event['away_team']; ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="betting-section">
                    <div class="main-markets">
                        <div class="market-group">
                            <h4 class="market-title">Match Result</h4>
                            <div class="odds-row">
                                <?php if ($event['home_odds']): ?>
                                <button class="odds-btn home" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'home', <?php echo $event['home_odds']; ?>)">
                                    <span class="label">1</span>
                                    <span class="odds"><?php echo number_format($event['home_odds'], 2); ?></span>
                                </button>
                                <?php endif; ?>
                                
                                <?php if ($event['draw_odds']): ?>
                                <button class="odds-btn draw" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'draw', <?php echo $event['draw_odds']; ?>)">
                                    <span class="label">X</span>
                                    <span class="odds"><?php echo number_format($event['draw_odds'], 2); ?></span>
                                </button>
                                <?php endif; ?>
                                
                                <?php if ($event['away_odds']): ?>
                                <button class="odds-btn away" onclick="addToBetSlip(<?php echo $event['id']; ?>, 'away', <?php echo $event['away_odds']; ?>)">
                                    <span class="label">2</span>
                                    <span class="odds"><?php echo number_format($event['away_odds'], 2); ?></span>
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="more-markets">
                            <button class="btn btn-outline btn-sm" onclick="showAllMarkets(<?php echo $event['id']; ?>)">
                                +<?php echo $event['total_markets'] ?? 50; ?> Markets
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</section>

<script>
document.addEventListener('DOMContentLoaded', function() {
    initializeLiveBetting();
    
    // Auto refresh if enabled
    const autoRefresh = document.getElementById('autoRefresh');
    if (autoRefresh && autoRefresh.checked) {
        setInterval(refreshLiveData, 15000); // Every 15 seconds
    }
});

function initializeLiveBetting() {
    // Initialize live betting functionality
    updateMatchTimes();
    setInterval(updateMatchTimes, 60000); // Update every minute
}

function refreshLiveData() {
    Crashify.api.request('/api/live-events/refresh')
        .then(response => {
            if (response.success) {
                updateLiveEvents(response.events);
                document.getElementById('lastUpdate').textContent = new Date().toLocaleTimeString();
            }
        })
        .catch(error => {
            console.error('Failed to refresh live data:', error);
        });
}

function updateMatchTimes() {
    const timeElements = document.querySelectorAll('.match-time');
    timeElements.forEach(element => {
        // Update match time logic would go here
    });
}

function openStream(eventId) {
    // Open live stream modal
    Crashify.ui.showModal('streamModal');
}

function showMatchStats(eventId) {
    // Show match statistics
    Crashify.api.request(`/api/events/${eventId}/stats`)
        .then(response => {
            if (response.success) {
                // Display stats
            }
        });
}

function showAllMarkets(eventId) {
    // Show all betting markets for event
    window.location.href = `/pages/event-details.php?id=${eventId}`;
}
</script>

<?php include '../includes/footer.php'; ?>
