<?php
// pages/support.php - Support Page
// صفحه پشتیبانی

require_once '../includes/auth.php';
require_once '../includes/functions.php';

$page_title = 'پشتیبانی';
$current_page = 'support';

// Handle form submissions
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_ticket'])) {
        $subject = sanitizeInput($_POST['subject']);
        $category = sanitizeInput($_POST['category']);
        $priority = sanitizeInput($_POST['priority']);
        $message = sanitizeInput($_POST['message']);
        
        if (empty($subject) || empty($message)) {
            $error_message = 'لطفاً تمام فیلدهای ضروری را پر کنید';
        } else {
            // Create support ticket
            $ticket_id = 'TK' . date('Ymd') . rand(1000, 9999);
            
            $stmt = $pdo->prepare("
                INSERT INTO support_tickets (ticket_id, user_id, subject, category, priority, status, created_at)
                VALUES (?, ?, ?, ?, ?, 'open', NOW())
            ");
            
            if ($stmt->execute([$ticket_id, $_SESSION['user_id'] ?? null, $subject, $category, $priority])) {
                $support_ticket_id = $pdo->lastInsertId();
                
                // Add initial message
                $stmt = $pdo->prepare("
                    INSERT INTO support_messages (ticket_id, user_id, message, is_admin, created_at)
                    VALUES (?, ?, ?, 0, NOW())
                ");
                
                if ($stmt->execute([$support_ticket_id, $_SESSION['user_id'] ?? null, $message])) {
                    $success_message = "تیکت شما با شماره $ticket_id ایجاد شد";
                } else {
                    $error_message = 'خطا در ارسال پیام';
                }
            } else {
                $error_message = 'خطا در ایجاد تیکت';
            }
        }
    }
    
    if (isset($_POST['send_message'])) {
        $ticket_id = intval($_POST['ticket_id']);
        $message = sanitizeInput($_POST['message']);
        
        if (empty($message)) {
            $error_message = 'لطفاً پیام خود را وارد کنید';
        } else {
            // Verify ticket belongs to user
            $stmt = $pdo->prepare("SELECT id FROM support_tickets WHERE id = ? AND (user_id = ? OR user_id IS NULL)");
            $stmt->execute([$ticket_id, $_SESSION['user_id'] ?? null]);
            
            if ($stmt->rowCount() > 0) {
                // Add message to ticket
                $stmt = $pdo->prepare("
                    INSERT INTO support_messages (ticket_id, user_id, message, is_admin, created_at)
                    VALUES (?, ?, ?, 0, NOW())
                ");
                
                if ($stmt->execute([$ticket_id, $_SESSION['user_id'] ?? null, $message])) {
                    // Update ticket status to 'waiting_admin'
                    $stmt = $pdo->prepare("UPDATE support_tickets SET status = 'waiting_admin', updated_at = NOW() WHERE id = ?");
                    $stmt->execute([$ticket_id]);
                    
                    $success_message = 'پیام شما ارسال شد';
                } else {
                    $error_message = 'خطا در ارسال پیام';
                }
            } else {
                $error_message = 'تیکت یافت نشد';
            }
        }
    }
}

// Get user tickets if logged in
$user_tickets = [];
if (isLoggedIn()) {
    $stmt = $pdo->prepare("
        SELECT st.*, 
               COUNT(sm.id) as message_count,
               MAX(sm.created_at) as last_message_at
        FROM support_tickets st
        LEFT JOIN support_messages sm ON st.id = sm.ticket_id
        WHERE st.user_id = ?
        GROUP BY st.id
        ORDER BY st.created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $user_tickets = $stmt->fetchAll();
}

// Get FAQ items
$stmt = $pdo->prepare("SELECT * FROM faq_items WHERE is_active = 1 ORDER BY sort_order ASC, id ASC");
$stmt->execute();
$faq_items = $stmt->fetchAll();

// Get current ticket if viewing one
$current_ticket = null;
$ticket_messages = [];
if (isset($_GET['ticket']) && isLoggedIn()) {
    $ticket_id = intval($_GET['ticket']);
    
    $stmt = $pdo->prepare("SELECT * FROM support_tickets WHERE id = ? AND user_id = ?");
    $stmt->execute([$ticket_id, $_SESSION['user_id']]);
    $current_ticket = $stmt->fetch();
    
    if ($current_ticket) {
        // Get ticket messages
        $stmt = $pdo->prepare("
            SELECT sm.*, u.first_name, u.last_name, u.avatar
            FROM support_messages sm
            LEFT JOIN users u ON sm.user_id = u.id
            WHERE sm.ticket_id = ?
            ORDER BY sm.created_at ASC
        ");
        $stmt->execute([$ticket_id]);
        $ticket_messages = $stmt->fetchAll();
        
        // Mark ticket as read by user
        $stmt = $pdo->prepare("UPDATE support_tickets SET user_read_at = NOW() WHERE id = ?");
        $stmt->execute([$ticket_id]);
    }
}

include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="page-header">
                <h1><i class="fas fa-headset"></i> پشتیبانی</h1>
                <p class="text-muted">ما اینجا هستیم تا به شما کمک کنیم</p>
            </div>
        </div>
    </div>
    
    <?php if ($success_message): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <?php echo $success_message; ?>
            <button type="button" class="close" data-dismiss="alert">&times;</button>
        </div>
    <?php endif; ?>
    
    <?php if ($error_message): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <?php echo $error_message; ?>
            <button type="button" class="close" data-dismiss="alert">&times;</button>
        </div>
    <?php endif; ?>
    
    <?php if ($current_ticket): ?>
        <!-- Ticket View -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <div>
                            <h5 class="mb-0">
                                <i class="fas fa-ticket-alt"></i> 
                                تیکت #<?php echo htmlspecialchars($current_ticket['ticket_id']); ?>
                            </h5>
                            <small class="text-muted"><?php echo htmlspecialchars($current_ticket['subject']); ?></small>
                        </div>
                        <div>
                            <span class="badge badge-<?php echo getStatusBadgeClass($current_ticket['status']); ?>">
                                <?php echo getStatusLabel($current_ticket['status']); ?>
                            </span>
                            <a href="/pages/support.php" class="btn btn-sm btn-outline-secondary ml-2">
                                <i class="fas fa-arrow-left"></i> بازگشت
                            </a>
                        </div>
                    </div>
                    
                    <div class="card-body">
                        <!-- Ticket Messages -->
                        <div class="ticket-messages mb-4">
                            <?php foreach ($ticket_messages as $message): ?>
                                <div class="message-item <?php echo $message['is_admin'] ? 'admin-message' : 'user-message'; ?>">
                                    <div class="message-avatar">
                                        <?php if ($message['avatar']): ?>
                                            <img src="/uploads/avatars/<?php echo htmlspecialchars($message['avatar']); ?>" 
                                                 alt="Avatar" class="rounded-circle" width="40" height="40">
                                        <?php else: ?>
                                            <div class="avatar-placeholder">
                                                <i class="fas fa-<?php echo $message['is_admin'] ? 'user-tie' : 'user'; ?>"></i>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="message-content">
                                        <div class="message-header">
                                            <strong>
                                                <?php if ($message['is_admin']): ?>
                                                    پشتیبانی
                                                <?php else: ?>
                                                    <?php echo htmlspecialchars($message['first_name'] . ' ' . $message['last_name']); ?>
                                                <?php endif; ?>
                                            </strong>
                                            <span class="message-time">
                                                <?php echo timeAgo($message['created_at']); ?>
                                            </span>
                                        </div>
                                        <div class="message-text">
                                            <?php echo nl2br(htmlspecialchars($message['message'])); ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <!-- Reply Form -->
                        <?php if ($current_ticket['status'] !== 'closed'): ?>
                            <form method="POST" class="reply-form">
                                <input type="hidden" name="ticket_id" value="<?php echo $current_ticket['id']; ?>">
                                <div class="form-group">
                                    <label for="message">پاسخ شما</label>
                                    <textarea class="form-control" id="message" name="message" rows="4" 
                                              placeholder="پیام خود را اینجا بنویسید..." required></textarea>
                                </div>
                                <button type="submit" name="send_message" class="btn btn-primary">
                                    <i class="fas fa-paper-plane"></i> ارسال پیام
                                </button>
                            </form>
                        <?php else: ?>
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i> این تیکت بسته شده است
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    <?php else: ?>
        <!-- Support Dashboard -->
        <div class="row">
            <!-- Quick Actions -->
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="card support-card">
                    <div class="card-body text-center">
                        <div class="support-icon">
                            <i class="fas fa-plus-circle text-primary"></i>
                        </div>
                        <h5>ایجاد تیکت جدید</h5>
                        <p class="text-muted">سوال یا مشکل جدید دارید؟</p>
                        <button class="btn btn-primary" data-toggle="modal" data-target="#newTicketModal">
                            <i class="fas fa-plus"></i> تیکت جدید
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="card support-card">
                    <div class="card-body text-center">
                        <div class="support-icon">
                            <i class="fas fa-comments text-success"></i>
                        </div>
                        <h5>چت آنلاین</h5>
                        <p class="text-muted">گفتگو مستقیم با پشتیبانی</p>
                        <button class="btn btn-success" onclick="startLiveChat()">
                            <i class="fas fa-comment"></i> شروع چت
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="card support-card">
                    <div class="card-body text-center">
                        <div class="support-icon">
                            <i class="fas fa-question-circle text-info"></i>
                        </div>
                        <h5>سوالات متداول</h5>
                        <p class="text-muted">پاسخ سوالات رایج</p>
                        <a href="#faq-section" class="btn btn-info">
                            <i class="fas fa-search"></i> مشاهده FAQ
                        </a>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- User Tickets -->
        <?php if (isLoggedIn() && !empty($user_tickets)): ?>
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h5><i class="fas fa-ticket-alt"></i> تیکت‌های شما</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>شماره تیکت</th>
                                            <th>موضوع</th>
                                            <th>دسته‌بندی</th>
                                            <th>وضعیت</th>
                                            <th>آخرین پیام</th>
                                            <th>عملیات</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($user_tickets as $ticket): ?>
                                            <tr>
                                                <td>
                                                    <strong>#<?php echo htmlspecialchars($ticket['ticket_id']); ?></strong>
                                                </td>
                                                <td><?php echo htmlspecialchars($ticket['subject']); ?></td>
                                                <td>
                                                    <span class="badge badge-secondary">
                                                        <?php echo getCategoryLabel($ticket['category']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php echo getStatusBadgeClass($ticket['status']); ?>">
                                                        <?php echo getStatusLabel($ticket['status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <?php if ($ticket['last_message_at']): ?>
                                                        <?php echo timeAgo($ticket['last_message_at']); ?>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <a href="/pages/support.php?ticket=<?php echo $ticket['id']; ?>" 
                                                       class="btn btn-sm btn-outline-primary">
                                                        <i class="fas fa-eye"></i> مشاهده
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- FAQ Section -->
        <div class="row" id="faq-section">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-question-circle"></i> سوالات متداول</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($faq_items)): ?>
                            <p class="text-muted text-center">سوالی یافت نشد</p>
                        <?php else: ?>
                            <div class="accordion" id="faqAccordion">
                                <?php foreach ($faq_items as $index => $faq): ?>
                                    <div class="card">
                                        <div class="card-header" id="faq<?php echo $index; ?>">
                                            <h6 class="mb-0">
                                                <button class="btn btn-link collapsed" type="button" 
                                                        data-toggle="collapse" data-target="#collapse<?php echo $index; ?>" 
                                                        aria-expanded="false" aria-controls="collapse<?php echo $index; ?>">
                                                    <?php echo htmlspecialchars($faq['question']); ?>
                                                </button>
                                            </h6>
                                        </div>
                                        <div id="collapse<?php echo $index; ?>" class="collapse" 
                                             aria-labelledby="faq<?php echo $index; ?>" data-parent="#faqAccordion">
                                            <div class="card-body">
                                                <?php echo nl2br(htmlspecialchars($faq['answer'])); ?>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Contact Information -->
        <div class="row">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-phone"></i> اطلاعات تماس</h5>
                    </div>
                    <div class="card-body">
                        <div class="contact-item">
                            <i class="fas fa-envelope text-primary"></i>
                            <div class="contact-info">
                                <strong>ایمیل پشتیبانی</strong>
                                <p>support@crashify.com</p>
                            </div>
                        </div>
                        
                        <div class="contact-item">
                            <i class="fas fa-phone text-success"></i>
                            <div class="contact-info">
                                <strong>تلفن پشتیبانی</strong>
                                <p>+1 (555) 123-4567</p>
                            </div>
                        </div>
                        
                        <div class="contact-item">
                            <i class="fas fa-clock text-info"></i>
                            <div class="contact-info">
                                <strong>ساعات کاری</strong>
                                <p>24/7 - همه روزه</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-info-circle"></i> راهنمای سریع</h5>
                    </div>
                    <div class="card-body">
                        <div class="quick-guide">
                            <div class="guide-item">
                                <i class="fas fa-user-plus text-primary"></i>
                                <div>
                                    <strong>ثبت نام</strong>
                                    <p>نحوه ایجاد حساب کاربری</p>
                                </div>
                            </div>
                            
                            <div class="guide-item">
                                <i class="fas fa-credit-card text-success"></i>
                                <div>
                                    <strong>واریز و برداشت</strong>
                                    <p>راهنمای تراکنش‌های مالی</p>
                                </div>
                            </div>
                            
                            <div class="guide-item">
                                <i class="fas fa-dice text-warning"></i>
                                <div>
                                    <strong>نحوه بازی</strong>
                                    <p>آموزش بازی‌های مختلف</p>
                                </div>
                            </div>
                            
                            <div class="guide-item">
                                <i class="fas fa-shield-alt text-danger"></i>
                                <div>
                                    <strong>امنیت حساب</strong>
                                    <p>تنظیمات امنیتی</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- New Ticket Modal -->
<div class="modal fade" id="newTicketModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-plus-circle"></i> ایجاد تیکت جدید
                </h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="subject">موضوع *</label>
                        <input type="text" class="form-control" id="subject" name="subject" 
                               placeholder="موضوع تیکت خود را وارد کنید" required>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="category">دسته‌بندی</label>
                                <select class="form-control" id="category" name="category">
                                    <option value="general">عمومی</option>
                                    <option value="account">حساب کاربری</option>
                                    <option value="payment">پرداخت</option>
                                    <option value="technical">فنی</option>
                                    <option value="game">بازی</option>
                                    <option value="complaint">شکایت</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="priority">اولویت</label>
                                <select class="form-control" id="priority" name="priority">
                                    <option value="low">کم</option>
                                    <option value="medium" selected>متوسط</option>
                                    <option value="high">بالا</option>
                                    <option value="urgent">فوری</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="message">پیام *</label>
                        <textarea class="form-control" id="message" name="message" rows="6" 
                                  placeholder="توضیح کاملی از مشکل یا سوال خود ارائه دهید..." required></textarea>
                    </div>
                    
                    <?php if (!isLoggedIn()): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            برای ایجاد تیکت، لطفاً ابتدا وارد حساب کاربری خود شوید
                        </div>
                    <?php endif; ?>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">انصراف</button>
                    <button type="submit" name="create_ticket" class="btn btn-primary" 
                            <?php echo !isLoggedIn() ? 'disabled' : ''; ?>>
                        <i class="fas fa-paper-plane"></i> ارسال تیکت
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.support-card {
    transition: transform 0.2s;
    height: 100%;
}

.support-card:hover {
    transform: translateY(-5px);
}

.support-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
}

.ticket-messages {
    max-height: 500px;
    overflow-y: auto;
}

.message-item {
    display: flex;
    margin-bottom: 1.5rem;
    padding: 1rem;
    border-radius: 8px;
}

.user-message {
    background: #f8f9fa;
    margin-left: 2rem;
}

.admin-message {
    background: #e3f2fd;
    margin-right: 2rem;
}

.message-avatar {
    margin-right: 1rem;
    flex-shrink: 0;
}

.avatar-placeholder {
    width: 40px;
    height: 40px;
    background: #dee2e6;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
}

.message-content {
    flex: 1;
}

.message-header {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 0.5rem;
}

.message-time {
    font-size: 0.875rem;
    color: #6c757d;
    margin-left: auto;
}

.message-text {
    line-height: 1.5;
}

.contact-item {
    display: flex;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #f8f9fa;
}

.contact-item:last-child {
    border-bottom: none;
    margin-bottom: 0;
    padding-bottom: 0;
}

.contact-item i {
    font-size: 1.5rem;
    margin-right: 1rem;
    width: 30px;
}

.contact-info strong {
    display: block;
    margin-bottom: 0.25rem;
}

.contact-info p {
    margin: 0;
    color: #6c757d;
}

.guide-item {
    display: flex;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #f8f9fa;
}

.guide-item:last-child {
    border-bottom: none;
    margin-bottom: 0;
    padding-bottom: 0;
}

.guide-item i {
    font-size: 1.5rem;
    margin-right: 1rem;
    width: 30px;
}

.guide-item strong {
    display: block;
    margin-bottom: 0.25rem;
}

.guide-item p {
    margin: 0;
    color: #6c757d;
    font-size: 0.875rem;
}

.page-header {
    margin-bottom: 2rem;
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin-bottom: 0.5rem;
}
</style>

<script>
function startLiveChat() {
    // Implement live chat functionality
    alert('چت آنلاین به زودی راه‌اندازی خواهد شد');
}

// Auto-resize textarea
document.addEventListener('DOMContentLoaded', function() {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        textarea.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = this.scrollHeight + 'px';
        });
    });
});
</script>

<?php
// Helper functions
function getStatusLabel($status) {
    $labels = [
        'open' => 'باز',
        'waiting_admin' => 'در انتظار پاسخ',
        'waiting_user' => 'در انتظار شما',
        'closed' => 'بسته شده',
        'resolved' => 'حل شده'
    ];
    
    return $labels[$status] ?? $status;
}

function getStatusBadgeClass($status) {
    $classes = [
        'open' => 'primary',
        'waiting_admin' => 'warning',
        'waiting_user' => 'info',
        'closed' => 'secondary',
        'resolved' => 'success'
    ];
    
    return $classes[$status] ?? 'secondary';
}

function getCategoryLabel($category) {
    $labels = [
        'general' => 'عمومی',
        'account' => 'حساب کاربری',
        'payment' => 'پرداخت',
        'technical' => 'فنی',
        'game' => 'بازی',
        'complaint' => 'شکایت'
    ];
    
    return $labels[$category] ?? $category;
}

include '../includes/footer.php';
?>