<?php
// pages/bonuses.php - Bonuses and Promotions Page
// صفحه جوایز و تخفیف‌ها

require_once '../includes/auth.php';
require_once '../includes/functions.php';

$page_title = 'جوایز و تخفیف‌ها';
$current_page = 'bonuses';

// Handle bonus claim
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['claim_bonus'])) {
    if (!isLoggedIn()) {
        $error_message = 'برای دریافت جایزه باید وارد حساب کاربری شوید';
    } else {
        $bonus_id = intval($_POST['bonus_id']);
        $user_id = $_SESSION['user_id'];
        
        // Check if bonus exists and is active
        $stmt = $pdo->prepare("SELECT * FROM bonuses WHERE id = ? AND is_active = 1 AND start_date <= NOW() AND end_date >= NOW()");
        $stmt->execute([$bonus_id]);
        $bonus = $stmt->fetch();
        
        if (!$bonus) {
            $error_message = 'جایزه یافت نشد یا منقضی شده است';
        } else {
            // Check if user already claimed this bonus
            $stmt = $pdo->prepare("SELECT id FROM user_bonuses WHERE user_id = ? AND bonus_id = ?");
            $stmt->execute([$user_id, $bonus_id]);
            
            if ($stmt->rowCount() > 0) {
                $error_message = 'شما قبلاً این جایزه را دریافت کرده‌اید';
            } else {
                // Check bonus requirements
                $can_claim = true;
                $requirements_error = '';
                
                if ($bonus['min_deposit'] > 0) {
                    $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount), 0) as total_deposits FROM transactions WHERE user_id = ? AND type = 'deposit' AND status = 'completed'");
                    $stmt->execute([$user_id]);
                    $total_deposits = $stmt->fetchColumn();
                    
                    if ($total_deposits < $bonus['min_deposit']) {
                        $can_claim = false;
                        $requirements_error = "حداقل واریز مورد نیاز: $" . number_format($bonus['min_deposit'], 2);
                    }
                }
                
                if ($bonus['min_bets'] > 0) {
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM bets WHERE user_id = ?");
                    $stmt->execute([$user_id]);
                    $total_bets = $stmt->fetchColumn();
                    
                    if ($total_bets < $bonus['min_bets']) {
                        $can_claim = false;
                        $requirements_error = "حداقل تعداد شرط مورد نیاز: " . $bonus['min_bets'];
                    }
                }
                
                if (!$can_claim) {
                    $error_message = 'شرایط دریافت جایزه را ندارید. ' . $requirements_error;
                } else {
                    // Calculate bonus amount
                    $bonus_amount = 0;
                    
                    if ($bonus['type'] === 'fixed') {
                        $bonus_amount = $bonus['amount'];
                    } elseif ($bonus['type'] === 'percentage') {
                        // Get user's last deposit for percentage bonus
                        $stmt = $pdo->prepare("SELECT amount FROM transactions WHERE user_id = ? AND type = 'deposit' AND status = 'completed' ORDER BY created_at DESC LIMIT 1");
                        $stmt->execute([$user_id]);
                        $last_deposit = $stmt->fetchColumn();
                        
                        if ($last_deposit) {
                            $bonus_amount = ($last_deposit * $bonus['percentage']) / 100;
                            if ($bonus['max_amount'] > 0) {
                                $bonus_amount = min($bonus_amount, $bonus['max_amount']);
                            }
                        }
                    }
                    
                    if ($bonus_amount > 0) {
                        try {
                            $pdo->beginTransaction();
                            
                            // Record bonus claim
                            $stmt = $pdo->prepare("
                                INSERT INTO user_bonuses (user_id, bonus_id, amount, status, claimed_at, expires_at)
                                VALUES (?, ?, ?, 'active', NOW(), DATE_ADD(NOW(), INTERVAL ? DAY))
                            ");
                            $stmt->execute([$user_id, $bonus_id, $bonus_amount, $bonus['validity_days'] ?: 30]);
                            
                            // Add bonus to user balance
                            $stmt = $pdo->prepare("UPDATE users SET balance = balance + ?, bonus_balance = bonus_balance + ? WHERE id = ?");
                            $stmt->execute([$bonus_amount, $bonus_amount, $user_id]);
                            
                            // Create transaction record
                            $stmt = $pdo->prepare("
                                INSERT INTO transactions (user_id, type, amount, status, description, created_at)
                                VALUES (?, 'bonus', ?, 'completed', ?, NOW())
                            ");
                            $stmt->execute([$user_id, $bonus_amount, 'دریافت جایزه: ' . $bonus['title']]);
                            
                            $pdo->commit();
                            $success_message = "جایزه با موفقیت دریافت شد! مبلغ $" . number_format($bonus_amount, 2) . " به حساب شما اضافه شد";
                            
                        } catch (Exception $e) {
                            $pdo->rollBack();
                            $error_message = 'خطا در پردازش جایزه';
                        }
                    } else {
                        $error_message = 'مبلغ جایزه محاسبه نشد';
                    }
                }
            }
        }
    }
}

// Get active bonuses
$stmt = $pdo->prepare("
    SELECT b.*, 
           CASE WHEN ub.id IS NOT NULL THEN 1 ELSE 0 END as is_claimed
    FROM bonuses b
    LEFT JOIN user_bonuses ub ON b.id = ub.bonus_id AND ub.user_id = ?
    WHERE b.is_active = 1 
    AND b.start_date <= NOW() 
    AND b.end_date >= NOW()
    ORDER BY b.sort_order ASC, b.created_at DESC
");
$stmt->execute([isLoggedIn() ? $_SESSION['user_id'] : 0]);
$bonuses = $stmt->fetchAll();

// Get user's active bonuses if logged in
$user_bonuses = [];
if (isLoggedIn()) {
    $stmt = $pdo->prepare("
        SELECT ub.*, b.title, b.description, b.type
        FROM user_bonuses ub
        JOIN bonuses b ON ub.bonus_id = b.id
        WHERE ub.user_id = ? AND ub.status = 'active' AND ub.expires_at > NOW()
        ORDER BY ub.claimed_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $user_bonuses = $stmt->fetchAll();
}

// Get bonus categories
$stmt = $pdo->prepare("SELECT DISTINCT category FROM bonuses WHERE is_active = 1 ORDER BY category");
$stmt->execute();
$categories = $stmt->fetchAll(PDO::FETCH_COLUMN);

include '../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="page-header text-center">
                <h1><i class="fas fa-gift"></i> جوایز و تخفیف‌ها</h1>
                <p class="text-muted">از جوایز و تخفیف‌های ویژه ما بهره‌مند شوید</p>
            </div>
        </div>
    </div>
    
    <?php if ($success_message): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="fas fa-check-circle"></i> <?php echo $success_message; ?>
            <button type="button" class="close" data-dismiss="alert">&times;</button>
        </div>
    <?php endif; ?>
    
    <?php if ($error_message): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="fas fa-exclamation-circle"></i> <?php echo $error_message; ?>
            <button type="button" class="close" data-dismiss="alert">&times;</button>
        </div>
    <?php endif; ?>
    
    <!-- User's Active Bonuses -->
    <?php if (isLoggedIn() && !empty($user_bonuses)): ?>
        <div class="row mb-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-star text-warning"></i> جوایز فعال شما</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <?php foreach ($user_bonuses as $user_bonus): ?>
                                <div class="col-md-6 col-lg-4 mb-3">
                                    <div class="active-bonus-card">
                                        <div class="bonus-header">
                                            <h6><?php echo htmlspecialchars($user_bonus['title']); ?></h6>
                                            <span class="bonus-amount">$<?php echo number_format($user_bonus['amount'], 2); ?></span>
                                        </div>
                                        <div class="bonus-details">
                                            <p class="bonus-description"><?php echo htmlspecialchars($user_bonus['description']); ?></p>
                                            <div class="bonus-expiry">
                                                <i class="fas fa-clock text-warning"></i>
                                                انقضا: <?php echo date('Y/m/d H:i', strtotime($user_bonus['expires_at'])); ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Bonus Categories Filter -->
    <?php if (!empty($categories)): ?>
        <div class="row mb-4">
            <div class="col-12">
                <div class="bonus-filters text-center">
                    <button class="btn btn-outline-primary active" data-category="all">همه</button>
                    <?php foreach ($categories as $category): ?>
                        <button class="btn btn-outline-primary" data-category="<?php echo htmlspecialchars($category); ?>">
                            <?php echo htmlspecialchars($category); ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- Available Bonuses -->
    <div class="row" id="bonuses-container">
        <?php if (empty($bonuses)): ?>
            <div class="col-12">
                <div class="text-center py-5">
                    <i class="fas fa-gift fa-3x text-muted mb-3"></i>
                    <h4 class="text-muted">در حال حاضر جایزه‌ای موجود نیست</h4>
                    <p class="text-muted">به زودی جوایز جدید اضافه خواهد شد</p>
                </div>
            </div>
        <?php else: ?>
            <?php foreach ($bonuses as $bonus): ?>
                <div class="col-md-6 col-lg-4 mb-4 bonus-item" data-category="<?php echo htmlspecialchars($bonus['category']); ?>">
                    <div class="bonus-card <?php echo $bonus['is_claimed'] ? 'claimed' : ''; ?>">
                        <div class="bonus-ribbon">
                            <?php if ($bonus['is_claimed']): ?>
                                <span class="ribbon claimed-ribbon">دریافت شده</span>
                            <?php elseif ($bonus['type'] === 'welcome'): ?>
                                <span class="ribbon welcome-ribbon">خوش‌آمدگویی</span>
                            <?php elseif ($bonus['type'] === 'deposit'): ?>
                                <span class="ribbon deposit-ribbon">واریز</span>
                            <?php elseif ($bonus['type'] === 'cashback'): ?>
                                <span class="ribbon cashback-ribbon">بازگشت وجه</span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="bonus-header">
                            <div class="bonus-icon">
                                <i class="fas fa-<?php echo getBonusIcon($bonus['type']); ?>"></i>
                            </div>
                            <h5 class="bonus-title"><?php echo htmlspecialchars($bonus['title']); ?></h5>
                        </div>
                        
                        <div class="bonus-body">
                            <div class="bonus-amount">
                                <?php if ($bonus['type'] === 'fixed'): ?>
                                    <span class="amount">$<?php echo number_format($bonus['amount'], 2); ?></span>
                                    <span class="amount-type">مبلغ ثابت</span>
                                <?php elseif ($bonus['type'] === 'percentage'): ?>
                                    <span class="amount"><?php echo $bonus['percentage']; ?>%</span>
                                    <span class="amount-type">درصد واریز</span>
                                    <?php if ($bonus['max_amount'] > 0): ?>
                                        <small class="text-muted d-block">حداکثر: $<?php echo number_format($bonus['max_amount'], 2); ?></small>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                            
                            <div class="bonus-description">
                                <p><?php echo nl2br(htmlspecialchars($bonus['description'])); ?></p>
                            </div>
                            
                            <div class="bonus-requirements">
                                <h6><i class="fas fa-list-check"></i> شرایط:</h6>
                                <ul class="requirements-list">
                                    <?php if ($bonus['min_deposit'] > 0): ?>
                                        <li>حداقل واریز: $<?php echo number_format($bonus['min_deposit'], 2); ?></li>
                                    <?php endif; ?>
                                    
                                    <?php if ($bonus['min_bets'] > 0): ?>
                                        <li>حداقل تعداد شرط: <?php echo $bonus['min_bets']; ?></li>
                                    <?php endif; ?>
                                    
                                    <?php if ($bonus['wagering_requirement'] > 0): ?>
                                        <li>ضریب گردش: <?php echo $bonus['wagering_requirement']; ?>x</li>
                                    <?php endif; ?>
                                    
                                    <?php if ($bonus['validity_days'] > 0): ?>
                                        <li>اعتبار: <?php echo $bonus['validity_days']; ?> روز</li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                            
                            <div class="bonus-validity">
                                <div class="validity-info">
                                    <i class="fas fa-calendar-alt"></i>
                                    <span>تا <?php echo date('Y/m/d', strtotime($bonus['end_date'])); ?></span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="bonus-footer">
                            <?php if ($bonus['is_claimed']): ?>
                                <button class="btn btn-secondary btn-block" disabled>
                                    <i class="fas fa-check"></i> دریافت شده
                                </button>
                            <?php elseif (!isLoggedIn()): ?>
                                <a href="/auth/login.php" class="btn btn-primary btn-block">
                                    <i class="fas fa-sign-in-alt"></i> ورود برای دریافت
                                </a>
                            <?php else: ?>
                                <form method="POST" class="claim-bonus-form">
                                    <input type="hidden" name="bonus_id" value="<?php echo $bonus['id']; ?>">
                                    <button type="submit" name="claim_bonus" class="btn btn-success btn-block">
                                        <i class="fas fa-gift"></i> دریافت جایزه
                                    </button>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <!-- Bonus Terms and Conditions -->
    <div class="row mt-5">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-file-contract"></i> قوانین و مقررات جوایز</h5>
                </div>
                <div class="card-body">
                    <div class="terms-content">
                        <h6>شرایط عمومی:</h6>
                        <ul>
                            <li>هر کاربر فقط یک بار می‌تواند از هر جایزه استفاده کند</li>
                            <li>جوایز دارای تاریخ انقضا هستند و پس از آن غیرقابل استفاده خواهند بود</li>
                            <li>برای برداشت جوایز، باید شرایط گردش (Wagering) را تکمیل کنید</li>
                            <li>سایت حق لغو یا تغییر جوایز را در هر زمان محفوظ می‌دارد</li>
                            <li>سوء استفاده از سیستم جوایز منجر به مسدود شدن حساب کاربری خواهد شد</li>
                        </ul>
                        
                        <h6>نحوه محاسبه ضریب گردش:</h6>
                        <p>
                            ضریب گردش به معنای تعداد دفعاتی است که باید مبلغ جایزه را شرط‌بندی کنید تا بتوانید آن را برداشت کنید.
                            مثال: اگر جایزه $100 با ضریب گردش 5x دریافت کنید، باید $500 شرط‌بندی کنید.
                        </p>
                        
                        <h6>بازی‌های مجاز:</h6>
                        <p>
                            تمام بازی‌های موجود در سایت برای تکمیل شرایط گردش مجاز هستند، اما درصد مشارکت آن‌ها متفاوت است:
                        </p>
                        <ul>
                            <li>بازی‌های کرش: 100%</li>
                            <li>اسلات‌ها: 100%</li>
                            <li>شرط‌بندی ورزشی: 50%</li>
                            <li>بازی‌های رومیزی: 10%</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.page-header {
    margin-bottom: 2rem;
    padding: 2rem 0;
}

.page-header h1 {
    color: #2c3e50;
    margin-bottom: 0.5rem;
}

.bonus-filters {
    margin-bottom: 2rem;
}

.bonus-filters .btn {
    margin: 0 0.25rem 0.5rem;
    border-radius: 25px;
}

.bonus-filters .btn.active {
    background-color: #007bff;
    border-color: #007bff;
    color: white;
}

.bonus-card {
    background: white;
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    transition: transform 0.3s, box-shadow 0.3s;
    position: relative;
    overflow: hidden;
    height: 100%;
    display: flex;
    flex-direction: column;
}

.bonus-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
}

.bonus-card.claimed {
    opacity: 0.7;
    background: #f8f9fa;
}

.bonus-ribbon {
    position: absolute;
    top: 15px;
    right: -30px;
    z-index: 2;
}

.ribbon {
    background: linear-gradient(45deg, #ff6b6b, #ee5a52);
    color: white;
    padding: 5px 40px;
    font-size: 0.75rem;
    font-weight: bold;
    transform: rotate(45deg);
    text-align: center;
    box-shadow: 0 2px 5px rgba(0,0,0,0.2);
}

.welcome-ribbon {
    background: linear-gradient(45deg, #4ecdc4, #44a08d);
}

.deposit-ribbon {
    background: linear-gradient(45deg, #667eea, #764ba2);
}

.cashback-ribbon {
    background: linear-gradient(45deg, #f093fb, #f5576c);
}

.claimed-ribbon {
    background: linear-gradient(45deg, #95a5a6, #7f8c8d);
}

.bonus-header {
    text-align: center;
    padding: 2rem 1.5rem 1rem;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.bonus-icon {
    font-size: 2.5rem;
    margin-bottom: 1rem;
}

.bonus-title {
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
}

.bonus-body {
    padding: 1.5rem;
    flex: 1;
    display: flex;
    flex-direction: column;
}

.bonus-amount {
    text-align: center;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 10px;
}

.bonus-amount .amount {
    display: block;
    font-size: 2rem;
    font-weight: bold;
    color: #28a745;
}

.bonus-amount .amount-type {
    font-size: 0.875rem;
    color: #6c757d;
}

.bonus-description {
    margin-bottom: 1.5rem;
    flex: 1;
}

.bonus-description p {
    color: #6c757d;
    line-height: 1.6;
}

.bonus-requirements {
    margin-bottom: 1.5rem;
}

.bonus-requirements h6 {
    color: #495057;
    margin-bottom: 0.75rem;
    font-size: 0.9rem;
}

.requirements-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.requirements-list li {
    padding: 0.25rem 0;
    font-size: 0.875rem;
    color: #6c757d;
    position: relative;
    padding-left: 1.5rem;
}

.requirements-list li:before {
    content: '✓';
    position: absolute;
    left: 0;
    color: #28a745;
    font-weight: bold;
}

.bonus-validity {
    margin-bottom: 1rem;
}

.validity-info {
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
    font-size: 0.875rem;
}

.validity-info i {
    margin-right: 0.5rem;
}

.bonus-footer {
    padding: 0 1.5rem 1.5rem;
    margin-top: auto;
}

.active-bonus-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 10px;
    padding: 1.5rem;
    height: 100%;
}

.active-bonus-card .bonus-header {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 1rem;
    padding: 0;
    background: none;
}

.active-bonus-card .bonus-amount {
    font-size: 1.25rem;
    font-weight: bold;
}

.active-bonus-card .bonus-description {
    margin-bottom: 1rem;
    opacity: 0.9;
}

.active-bonus-card .bonus-expiry {
    font-size: 0.875rem;
    opacity: 0.8;
}

.terms-content h6 {
    color: #495057;
    margin-top: 1.5rem;
    margin-bottom: 0.75rem;
}

.terms-content h6:first-child {
    margin-top: 0;
}

.terms-content ul {
    margin-bottom: 1.5rem;
}

.terms-content li {
    margin-bottom: 0.5rem;
    line-height: 1.6;
}

@media (max-width: 768px) {
    .bonus-filters .btn {
        font-size: 0.875rem;
        padding: 0.375rem 0.75rem;
    }
    
    .bonus-header {
        padding: 1.5rem 1rem 0.75rem;
    }
    
    .bonus-icon {
        font-size: 2rem;
    }
    
    .bonus-title {
        font-size: 1.1rem;
    }
    
    .bonus-amount .amount {
        font-size: 1.5rem;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Category filtering
    const filterButtons = document.querySelectorAll('[data-category]');
    const bonusItems = document.querySelectorAll('.bonus-item');
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            const category = this.dataset.category;
            
            // Update active button
            filterButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Filter bonus items
            bonusItems.forEach(item => {
                if (category === 'all' || item.dataset.category === category) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    });
    
    // Claim bonus form handling
    const claimForms = document.querySelectorAll('.claim-bonus-form');
    claimForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const button = this.querySelector('button[type="submit"]');
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> در حال پردازش...';
        });
    });
});
</script>

<?php
// Helper function to get bonus icon
function getBonusIcon($type) {
    $icons = [
        'welcome' => 'hand-holding-heart',
        'deposit' => 'coins',
        'cashback' => 'undo-alt',
        'loyalty' => 'crown',
        'referral' => 'users',
        'tournament' => 'trophy',
        'default' => 'gift'
    ];
    
    return $icons[$type] ?? $icons['default'];
}

include '../includes/footer.php';
?>