<?php
// includes/auth.php - Authentication System
// سیستم احراز هویت Crashify

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

class Auth {
    private static $instance = null;
    private $currentUser = null;
    private $sessionTimeout = 7200; // 2 hours
    
    private function __construct() {
        $this->checkSession();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function checkSession() {
        // Check if user is logged in
        if (!isset($_SESSION['user_id'])) {
            return false;
        }
        
        // Check session timeout
        if (isset($_SESSION['last_activity']) && 
            (time() - $_SESSION['last_activity']) > $this->sessionTimeout) {
            $this->logout();
            return false;
        }
        
        // Update last activity
        $_SESSION['last_activity'] = time();
        
        // Load current user
        $this->loadCurrentUser();
        
        return true;
    }
    
    private function loadCurrentUser() {
        if (!isset($_SESSION['user_id'])) {
            return;
        }
        
        try {
            $this->currentUser = fetchOne(
                "SELECT u.*, 
                        COUNT(DISTINCT b.id) as total_bets,
                        COUNT(DISTINCT CASE WHEN b.status = 'won' THEN b.id END) as total_wins,
                        COUNT(DISTINCT CASE WHEN b.status = 'lost' THEN b.id END) as total_losses,
                        COALESCE(SUM(CASE WHEN b.status = 'won' THEN b.actual_payout ELSE 0 END), 0) as total_winnings,
                        COALESCE(SUM(b.stake), 0) as total_wagered,
                        COUNT(DISTINCT ub.id) as active_bonuses,
                        COUNT(DISTINCT CASE WHEN n.read_at IS NULL THEN n.id END) as unread_notifications
                 FROM users u 
                 LEFT JOIN bets b ON u.id = b.user_id 
                 LEFT JOIN user_bonuses ub ON u.id = ub.user_id AND ub.status = 'active'
                 LEFT JOIN notifications n ON u.id = n.user_id
                 WHERE u.id = ? AND u.status = 'active'
                 GROUP BY u.id",
                [$_SESSION['user_id']]
            );
            
            if (!$this->currentUser) {
                $this->logout();
                return;
            }
            
            // Calculate win rate
            $this->currentUser['win_rate'] = $this->currentUser['total_bets'] > 0 ? 
                ($this->currentUser['total_wins'] / $this->currentUser['total_bets']) * 100 : 0;
            
            // Update user's last activity
            updateRecord('users', [
                'last_login' => date('Y-m-d H:i:s'),
                'last_ip' => $this->getClientIP()
            ], 'id = ?', [$this->currentUser['id']]);
            
        } catch (Exception $e) {
            error_log("Failed to load current user: " . $e->getMessage());
            $this->logout();
        }
    }
    
    public function login($identifier, $password, $rememberMe = false) {
        try {
            // Rate limiting
            $clientIP = $this->getClientIP();
            if (!$this->checkRateLimit('login_' . $clientIP, 5, 900)) { // 5 attempts per 15 minutes
                return [
                    'success' => false, 
                    'message' => 'Too many login attempts. Please try again later.',
                    'code' => 'RATE_LIMITED'
                ];
            }
            
            // Find user by email or username
            $user = fetchOne(
                "SELECT * FROM users WHERE (email = ? OR username = ?) AND status = 'active'",
                [$identifier, $identifier]
            );
            
            if (!$user) {
                $this->recordFailedAttempt($identifier, $clientIP, 'user_not_found');
                return [
                    'success' => false, 
                    'message' => 'Invalid email/username or password',
                    'code' => 'INVALID_CREDENTIALS'
                ];
            }
            
            // Verify password
            if (!password_verify($password, $user['password_hash'])) {
                $this->recordFailedAttempt($identifier, $clientIP, 'invalid_password', $user['id']);
                return [
                    'success' => false, 
                    'message' => 'Invalid email/username or password',
                    'code' => 'INVALID_CREDENTIALS'
                ];
            }
            
            // Check if account is locked
            if ($this->isAccountLocked($user['id'])) {
                return [
                    'success' => false, 
                    'message' => 'Account is temporarily locked due to multiple failed attempts',
                    'code' => 'ACCOUNT_LOCKED'
                ];
            }
            
            // Check 2FA if enabled
            if ($user['two_factor_enabled']) {
                $_SESSION['2fa_user_id'] = $user['id'];
                return [
                    'success' => false,
                    'message' => 'Two-factor authentication required',
                    'code' => 'REQUIRE_2FA',
                    'redirect' => '/auth/2fa.php'
                ];
            }
            
            // Login successful
            $this->createSession($user, $rememberMe);
            $this->clearFailedAttempts($user['id']);
            $this->logActivity($user['id'], 'login', ['ip' => $clientIP]);
            
            return [
                'success' => true,
                'message' => 'Login successful',
                'user' => $this->sanitizeUserData($user),
                'redirect' => $_SESSION['redirect_after_login'] ?? '/dashboard'
            ];
            
        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'An error occurred during login',
                'code' => 'SYSTEM_ERROR'
            ];
        }
    }
    
    public function register($userData) {
        try {
            // Validate input
            $validation = $this->validateRegistrationData($userData);
            if (!$validation['valid']) {
                return [
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validation['errors']
                ];
            }
            
            // Rate limiting
            $clientIP = $this->getClientIP();
            if (!$this->checkRateLimit('register_' . $clientIP, 3, 3600)) { // 3 registrations per hour
                return [
                    'success' => false,
                    'message' => 'Too many registration attempts. Please try again later.',
                    'code' => 'RATE_LIMITED'
                ];
            }
            
            // Check if user already exists
            $existingUser = fetchOne(
                "SELECT id FROM users WHERE email = ? OR username = ?",
                [$userData['email'], $userData['username']]
            );
            
            if ($existingUser) {
                return [
                    'success' => false,
                    'message' => 'User with this email or username already exists',
                    'code' => 'USER_EXISTS'
                ];
            }
            
            beginTransaction();
            
            // Create user account
            $userId = insertRecord('users', [
                'username' => $userData['username'],
                'email' => $userData['email'],
                'password_hash' => password_hash($userData['password'], PASSWORD_ARGON2ID, [
                    'memory_cost' => 65536,
                    'time_cost' => 4,
                    'threads' => 3
                ]),
                'first_name' => $userData['first_name'] ?? '',
                'last_name' => $userData['last_name'] ?? '',
                'country' => $userData['country'] ?? '',
                'date_of_birth' => $userData['date_of_birth'] ?? null,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            // Create email verification token
            $verificationToken = $this->generateSecureToken(64);
            insertRecord('email_verifications', [
                'user_id' => $userId,
                'token' => hash('sha256', $verificationToken),
                'expires_at' => date('Y-m-d H:i:s', strtotime('+24 hours'))
            ]);
            
            // Create welcome bonus
            $this->createWelcomeBonus($userId);
            
            // Send welcome email
            $this->sendWelcomeEmail($userData['email'], $userData['username'], $verificationToken);
            
            // Log activity
            $this->logActivity($userId, 'register', ['ip' => $clientIP]);
            
            commit();
            
            return [
                'success' => true,
                'message' => 'Registration successful! Please check your email to verify your account.',
                'user_id' => $userId
            ];
            
        } catch (Exception $e) {
            rollback();
            error_log("Registration error: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'An error occurred during registration',
                'code' => 'SYSTEM_ERROR'
            ];
        }
    }
    
    public function verify2FA($code) {
        if (!isset($_SESSION['2fa_user_id'])) {
            return ['success' => false, 'message' => 'Invalid session'];
        }
        
        $user = fetchOne("SELECT * FROM users WHERE id = ?", [$_SESSION['2fa_user_id']]);
        if (!$user) {
            return ['success' => false, 'message' => 'User not found'];
        }
        
        // Verify TOTP code
        if (!$this->verifyTOTP($user['two_factor_secret'], $code)) {
            return ['success' => false, 'message' => 'Invalid verification code'];
        }
        
        // Complete login
        unset($_SESSION['2fa_user_id']);
        $this->createSession($user);
        
        return [
            'success' => true,
            'message' => 'Login successful',
            'redirect' => '/dashboard'
        ];
    }
    
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            // Log activity
            $this->logActivity($_SESSION['user_id'], 'logout');
            
            // Remove session from database
            deleteRecord('user_sessions', 'id = ?', [session_id()]);
            
            // Clear remember me token
            if (isset($_COOKIE['remember_token'])) {
                deleteRecord('remember_tokens', 'token = ?', [hash('sha256', $_COOKIE['remember_token'])]);
                setcookie('remember_token', '', time() - 3600, '/', '', true, true);
            }
        }
        
        // Clear session data
        session_unset();
        session_destroy();
        
        // Start new session
        session_start();
        session_regenerate_id(true);
        
        $this->currentUser = null;
    }
    
    public function getCurrentUser() {
        return $this->currentUser;
    }
    
    public function isLoggedIn() {
        return $this->currentUser !== null;
    }
    
    public function requireAuth() {
        if (!$this->isLoggedIn()) {
            $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
            header('Location: /auth/login.php');
            exit;
        }
    }
    
    public function requireGuest() {
        if ($this->isLoggedIn()) {
            header('Location: /dashboard');
            exit;
        }
    }
    
    private function createSession($user, $rememberMe = false) {
        // Regenerate session ID
        session_regenerate_id(true);
        
        // Set session variables
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();
        
        // Create session record
        insertRecord('user_sessions', [
            'id' => session_id(),
            'user_id' => $user['id'],
            'ip_address' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        // Handle remember me
        if ($rememberMe) {
            $token = $this->generateSecureToken(64);
            $hashedToken = hash('sha256', $token);
            
            insertRecord('remember_tokens', [
                'user_id' => $user['id'],
                'token' => $hashedToken,
                'expires_at' => date('Y-m-d H:i:s', strtotime('+30 days'))
            ]);
            
            setcookie('remember_token', $token, time() + (30 * 24 * 60 * 60), '/', '', true, true);
        }
        
        $this->currentUser = $user;
    }
    
    private function validateRegistrationData($data) {
        $errors = [];
        
        // Username validation
        if (empty($data['username'])) {
            $errors['username'] = 'Username is required';
        } elseif (strlen($data['username']) < 3 || strlen($data['username']) > 20) {
            $errors['username'] = 'Username must be between 3 and 20 characters';
        } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $data['username'])) {
            $errors['username'] = 'Username can only contain letters, numbers, and underscores';
        }
        
        // Email validation
        if (empty($data['email'])) {
            $errors['email'] = 'Email is required';
        } elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = 'Invalid email format';
        }
        
        // Password validation
        if (empty($data['password'])) {
            $errors['password'] = 'Password is required';
        } elseif (strlen($data['password']) < 8) {
            $errors['password'] = 'Password must be at least 8 characters long';
        } elseif (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/', $data['password'])) {
            $errors['password'] = 'Password must contain at least one uppercase letter, one lowercase letter, and one number';
        }
        
        // Confirm password
        if (empty($data['confirm_password'])) {
            $errors['confirm_password'] = 'Please confirm your password';
        } elseif ($data['password'] !== $data['confirm_password']) {
            $errors['confirm_password'] = 'Passwords do not match';
        }
        
        // Age verification
        if (!empty($data['date_of_birth'])) {
            $birthDate = new DateTime($data['date_of_birth']);
            $today = new DateTime();
            $age = $today->diff($birthDate)->y;
            
            if ($age < 18) {
                $errors['date_of_birth'] = 'You must be at least 18 years old to register';
            }
        }
        
        // Terms acceptance
        if (empty($data['accept_terms'])) {
            $errors['accept_terms'] = 'You must accept the terms and conditions';
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    private function isAccountLocked($userId) {
        $lockTime = fetchColumn(
            "SELECT locked_until FROM users WHERE id = ? AND locked_until > NOW()",
            [$userId]
        );
        
        return $lockTime !== false;
    }
    
    private function recordFailedAttempt($identifier, $ip, $reason, $userId = null) {
        insertRecord('login_attempts', [
            'user_id' => $userId,
            'identifier' => $identifier,
            'ip_address' => $ip,
            'reason' => $reason,
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        // Check if we should lock the account
        if ($userId) {
            $recentAttempts = fetchColumn(
                "SELECT COUNT(*) FROM login_attempts 
                 WHERE user_id = ? AND created_at > DATE_SUB(NOW(), INTERVAL 15 MINUTE)",
                [$userId]
            );
            
            if ($recentAttempts >= 5) {
                updateRecord('users', [
                    'locked_until' => date('Y-m-d H:i:s', strtotime('+30 minutes'))
                ], 'id = ?', [$userId]);
            }
        }
    }
    
    private function clearFailedAttempts($userId) {
        deleteRecord('login_attempts', 'user_id = ?', [$userId]);
        updateRecord('users', ['locked_until' => null], 'id = ?', [$userId]);
    }
    
    private function checkRateLimit($key, $maxAttempts, $timeWindow) {
        $cacheKey = "rate_limit:{$key}";
        $attempts = apcu_fetch($cacheKey) ?: 0;
        
        if ($attempts >= $maxAttempts) {
            return false;
        }
        
        apcu_store($cacheKey, $attempts + 1, $timeWindow);
        return true;
    }
    
    private function generateSecureToken($length = 32) {
        return bin2hex(random_bytes($length));
    }
    
    private function getClientIP() {
        $ipKeys = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        
        foreach ($ipKeys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    private function sanitizeUserData($user) {
        unset($user['password_hash']);
        unset($user['two_factor_secret']);
        return $user;
    }
    
    private function logActivity($userId, $action, $details = []) {
        insertRecord('activity_logs', [
            'user_id' => $userId,
            'action' => $action,
            'details' => json_encode($details),
            'ip_address' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'created_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    private function createWelcomeBonus($userId) {
        $welcomeBonus = fetchOne(
            "SELECT * FROM bonuses WHERE type = 'welcome' AND active = 1 ORDER BY id DESC LIMIT 1"
        );
        
        if ($welcomeBonus) {
            $amount = $welcomeBonus['amount'];
            $wageringRequirement = $amount * $welcomeBonus['wagering_requirement'];
            $expiresAt = date('Y-m-d H:i:s', strtotime("+{$welcomeBonus['valid_days']} days"));
            
            insertRecord('user_bonuses', [
                'user_id' => $userId,
                'bonus_id' => $welcomeBonus['id'],
                'amount' => $amount,
                'wagering_requirement' => $wageringRequirement,
                'expires_at' => $expiresAt
            ]);
            
            // Add to user bonus balance
            updateRecord('users', [
                'bonus_balance' => $amount
            ], 'id = ?', [$userId]);
        }
    }
    
    private function sendWelcomeEmail($email, $username, $verificationToken) {
        // Email sending logic would go here
        // For now, just log it
        error_log("Welcome email sent to {$email} with verification token");
    }
    
    private function verifyTOTP($secret, $code) {
        // TOTP verification logic
        // This would use a library like RobThree/TwoFactorAuth
        return true; // Placeholder
    }
}

// Global auth instance
$auth = Auth::getInstance();
$currentUser = $auth->getCurrentUser();

// Helper functions
function requireAuth() {
    global $auth;
    $auth->requireAuth();
}

function requireGuest() {
    global $auth;
    $auth->requireGuest();
}

function isLoggedIn() {
    global $auth;
    return $auth->isLoggedIn();
}

function getCurrentUser() {
    global $currentUser;
    return $currentUser;
}
?>
