// assets/js/main.js - Main JavaScript File
// فایل جاوا اسکریپت اصلی برای سایت Crashify

'use strict';

// Global Configuration
window.Crashify = {
    config: {
        apiUrl: '/api',
        wsUrl: 'wss://ws.crashify.com',
        version: '1.0.0',
        debug: false,
        language: 'en',
        theme: 'light',
        currency: 'USD',
        timezone: 'UTC'
    },
    
    // State Management
    state: {
        user: null,
        isLoggedIn: false,
        notifications: [],
        betSlip: [],
        liveUpdates: true,
        websocket: null,
        modals: new Map(),
        dropdowns: new Map()
    },
    
    // Event Emitter
    events: new EventTarget(),
    
    // Utility Functions
    utils: {},
    
    // API Functions
    api: {},
    
    // UI Components
    ui: {},
    
    // Modules
    modules: {}
};

// Utility Functions
Crashify.utils = {
    // DOM Utilities
    $(selector, context = document) {
        return context.querySelector(selector);
    },
    
    $$(selector, context = document) {
        return Array.from(context.querySelectorAll(selector));
    },
    
    createElement(tag, attributes = {}, children = []) {
        const element = document.createElement(tag);
        
        Object.entries(attributes).forEach(([key, value]) => {
            if (key === 'className') {
                element.className = value;
            } else if (key === 'innerHTML') {
                element.innerHTML = value;
            } else if (key.startsWith('data-')) {
                element.setAttribute(key, value);
            } else {
                element[key] = value;
            }
        });
        
        children.forEach(child => {
            if (typeof child === 'string') {
                element.appendChild(document.createTextNode(child));
            } else if (child instanceof Node) {
                element.appendChild(child);
            }
        });
        
        return element;
    },
    
    // Event Utilities
    on(element, event, handler, options = {}) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            element.addEventListener(event, handler, options);
        }
    },
    
    off(element, event, handler) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            element.removeEventListener(event, handler);
        }
    },
    
    trigger(element, event, data = {}) {
        if (typeof element === 'string') {
            element = this.$(element);
        }
        if (element) {
            const customEvent = new CustomEvent(event, { detail: data });
            element.dispatchEvent(customEvent);
        }
    },
    
    // Animation Utilities
    fadeIn(element, duration = 300) {
        element.style.opacity = '0';
        element.style.display = 'block';
        
        const start = performance.now();
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.opacity = progress;
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    fadeOut(element, duration = 300) {
        const start = performance.now();
        const startOpacity = parseFloat(getComputedStyle(element).opacity);
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.opacity = startOpacity * (1 - progress);
            
            if (progress >= 1) {
                element.style.display = 'none';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    slideDown(element, duration = 300) {
        element.style.height = '0';
        element.style.overflow = 'hidden';
        element.style.display = 'block';
        
        const targetHeight = element.scrollHeight;
        const start = performance.now();
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.height = (targetHeight * progress) + 'px';
            
            if (progress >= 1) {
                element.style.height = '';
                element.style.overflow = '';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    slideUp(element, duration = 300) {
        const startHeight = element.offsetHeight;
        const start = performance.now();
        
        element.style.overflow = 'hidden';
        
        const animate = (currentTime) => {
            const elapsed = currentTime - start;
            const progress = Math.min(elapsed / duration, 1);
            
            element.style.height = (startHeight * (1 - progress)) + 'px';
            
            if (progress >= 1) {
                element.style.display = 'none';
                element.style.height = '';
                element.style.overflow = '';
            } else {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    },
    
    // Data Utilities
    formatCurrency(amount, currency = 'USD', decimals = 2) {
        const formatter = new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency,
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        });
        return formatter.format(amount);
    },
    
    formatNumber(number, decimals = 0) {
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals
        }).format(number);
    },
    
    formatDate(date, options = {}) {
        const defaultOptions = {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        };
        
        return new Intl.DateTimeFormat('en-US', { ...defaultOptions, ...options }).format(new Date(date));
    },
    
    formatTime(seconds) {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        
        if (hours > 0) {
            return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        }
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    },
    
    // Validation Utilities
    validateEmail(email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return regex.test(email);
    },
    
    validatePassword(password) {
        // At least 8 characters, 1 uppercase, 1 lowercase, 1 number
        const regex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/;
        return regex.test(password);
    },
    
    validateBetAmount(amount, min = 0.01, max = 10000) {
        const num = parseFloat(amount);
        return !isNaN(num) && num >= min && num <= max;
    },
    
    // Storage Utilities
    storage: {
        set(key, value, expiry = null) {
            const item = {
                value: value,
                expiry: expiry ? Date.now() + expiry : null
            };
            localStorage.setItem(key, JSON.stringify(item));
        },
        
        get(key) {
            const item = localStorage.getItem(key);
            if (!item) return null;
            
            try {
                const parsed = JSON.parse(item);
                if (parsed.expiry && Date.now() > parsed.expiry) {
                    localStorage.removeItem(key);
                    return null;
                }
                return parsed.value;
            } catch (e) {
                return null;
            }
        },
        
        remove(key) {
            localStorage.removeItem(key);
        },
        
        clear() {
            localStorage.clear();
        }
    },
    
    // URL Utilities
    getUrlParams() {
        return new URLSearchParams(window.location.search);
    },
    
    updateUrl(params, replace = false) {
        const url = new URL(window.location);
        Object.entries(params).forEach(([key, value]) => {
            if (value === null || value === undefined) {
                url.searchParams.delete(key);
            } else {
                url.searchParams.set(key, value);
            }
        });
        
        if (replace) {
            window.history.replaceState({}, '', url);
        } else {
            window.history.pushState({}, '', url);
        }
    },
    
    // Debounce and Throttle
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },
    
    throttle(func, limit) {
        let inThrottle;
        return function executedFunction(...args) {
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    },
    
    // Random Utilities
    generateId() {
        return Math.random().toString(36).substr(2, 9);
    },
    
    randomBetween(min, max) {
        return Math.random() * (max - min) + min;
    },
    
    // Device Detection
    isMobile() {
        return window.innerWidth <= 768;
    },
    
    isTablet() {
        return window.innerWidth > 768 && window.innerWidth <= 1024;
    },
    
    isDesktop() {
        return window.innerWidth > 1024;
    },
    
    // Copy to Clipboard
    async copyToClipboard(text) {
        try {
            await navigator.clipboard.writeText(text);
            return true;
        } catch (err) {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = text;
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                return true;
            } catch (e) {
                return false;
            } finally {
                document.body.removeChild(textArea);
            }
        }
    }
};

// API Functions
Crashify.api = {
    // Base request function
    async request(endpoint, options = {}) {
        const defaultOptions = {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            credentials: 'same-origin'
        };
        
        const config = { ...defaultOptions, ...options };
        
        // Add CSRF token if available
        const csrfToken = Crashify.utils.$('meta[name="csrf-token"]')?.getAttribute('content');
        if (csrfToken) {
            config.headers['X-CSRF-TOKEN'] = csrfToken;
        }
        
        try {
            const response = await fetch(`${Crashify.config.apiUrl}${endpoint}`, config);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return await response.json();
            }
            
            return await response.text();
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    },
    
    // Authentication
    async login(credentials) {
        return this.request('/auth/login', {
            method: 'POST',
            body: JSON.stringify(credentials)
        });
    },
    
    async register(userData) {
        return this.request('/auth/register', {
            method: 'POST',
            body: JSON.stringify(userData)
        });
    },
    
    async logout() {
        return this.request('/auth/logout', {
            method: 'POST'
        });
    },
    
    // User
    async getProfile() {
        return this.request('/user/profile');
    },
    
    async updateProfile(data) {
        return this.request('/user/profile', {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    },
    
    // Wallet
    async getBalance() {
        return this.request('/wallet/balance');
    },
    
    async deposit(data) {
        return this.request('/wallet/deposit', {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    async withdraw(data) {
        return this.request('/wallet/withdraw', {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    // Betting
    async placeBet(betData) {
        return this.request('/betting/place', {
            method: 'POST',
            body: JSON.stringify(betData)
        });
    },
    
    async getBetHistory(params = {}) {
        const queryString = new URLSearchParams(params).toString();
        return this.request(`/betting/history?${queryString}`);
    },
    
    // Sports
    async getSports() {
        return this.request('/sports');
    },
    
    async getEvents(sportId, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        return this.request(`/sports/${sportId}/events?${queryString}`);
    },
    
    async getOdds(eventId) {
        return this.request(`/events/${eventId}/odds`);
    },
    
    // Notifications
    async getNotifications() {
        return this.request('/notifications');
    },
    
    async markNotificationRead(id) {
        return this.request(`/notifications/${id}/read`, {
            method: 'POST'
        });
    }
};

// UI Components
Crashify.ui = {
    // Toast Notifications
    toast: {
        container: null,
        
        init() {
            if (!this.container) {
                this.container = Crashify.utils.createElement('div', {
                    className: 'toast-container',
                    style: 'position: fixed; top: 20px; right: 20px; z-index: 1080;'
                });
                document.body.appendChild(this.container);
            }
        },
        
        show(message, type = 'info', duration = 5000) {
            this.init();
            
            const toast = Crashify.utils.createElement('div', {
                className: `alert alert-${type} toast-item`,
                style: 'margin-bottom: 10px; min-width: 300px; opacity: 0; transform: translateX(100%); transition: all 0.3s ease;'
            });
            
            const content = Crashify.utils.createElement('div', {
                className: 'alert-content',
                innerHTML: `<i class="fas fa-${this.getIcon(type)}"></i><span>${message}</span>`
            });
            
            const closeBtn = Crashify.utils.createElement('button', {
                className: 'alert-close',
                innerHTML: '<i class="fas fa-times"></i>'
            });
            
            toast.appendChild(content);
            toast.appendChild(closeBtn);
            this.container.appendChild(toast);
            
            // Animate in
            requestAnimationFrame(() => {
                toast.style.opacity = '1';
                toast.style.transform = 'translateX(0)';
            });
            
            // Auto remove
            const timeoutId = setTimeout(() => {
                this.remove(toast);
            }, duration);
            
            // Manual close
            closeBtn.addEventListener('click', () => {
                clearTimeout(timeoutId);
                this.remove(toast);
            });
        },
        
        remove(toast) {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 300);
        },
        
        getIcon(type) {
            const icons = {
                success: 'check-circle',
                error: 'exclamation-circle',
                warning: 'exclamation-triangle',
                info: 'info-circle'
            };
            return icons[type] || 'info-circle';
        },
        
        success(message, duration) {
            this.show(message, 'success', duration);
        },
        
        error(message, duration) {
            this.show(message, 'error', duration);
        },
        
        warning(message, duration) {
            this.show(message, 'warning', duration);
        },
        
        info(message, duration) {
            this.show(message, 'info', duration);
        }
    },
    
    // Loading Overlay
    loading: {
        overlay: null,
        
        show(message = 'Loading...') {
            if (!this.overlay) {
                this.overlay = Crashify.utils.createElement('div', {
                    id: 'loadingOverlay',
                    className: 'loading-overlay',
                    innerHTML: `
                        <div class="loading-spinner">
                            <div class="spinner-ring"></div>
                            <div class="loading-text">${message}</div>
                        </div>
                    `
                });
                document.body.appendChild(this.overlay);
            }
            
            this.overlay.style.display = 'flex';
            Crashify.utils.fadeIn(this.overlay, 200);
        },
        
        hide() {
            if (this.overlay) {
                Crashify.utils.fadeOut(this.overlay, 200);
            }
        },
        
        updateMessage(message) {
            if (this.overlay) {
                const textElement = this.overlay.querySelector('.loading-text');
                if (textElement) {
                    textElement.textContent = message;
                }
            }
        }
    },
    
    // Confirmation Dialog
    confirm(message, title = 'Confirm', options = {}) {
        return new Promise((resolve) => {
            const modal = Crashify.utils.createElement('div', {
                className: 'modal-backdrop active'
            });
            
            const dialog = Crashify.utils.createElement('div', {
                className: 'modal active',
                style: 'max-width: 400px;',
                innerHTML: `
                    <div class="modal-header">
                        <h3 class="modal-title">${title}</h3>
                    </div>
                    <div class="modal-body">
                        <p>${message}</p>
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-outline cancel-btn">${options.cancelText || 'Cancel'}</button>
                        <button class="btn btn-primary confirm-btn">${options.confirmText || 'Confirm'}</button>
                    </div>
                `
            });
            
            modal.appendChild(dialog);
            document.body.appendChild(modal);
            
            const cleanup = () => {
                document.body.removeChild(modal);
            };
            
            dialog.querySelector('.cancel-btn').addEventListener('click', () => {
                cleanup();
                resolve(false);
            });
            
            dialog.querySelector('.confirm-btn').addEventListener('click', () => {
                cleanup();
                resolve(true);
            });
            
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    cleanup();
                    resolve(false);
                }
            });
        });
    }
};

// Theme Management
function initializeTheme(theme = 'light') {
    Crashify.config.theme = theme;
    document.body.className = document.body.className.replace(/theme-\w+/g, '');
    document.body.classList.add(`theme-${theme}`);
    
    // Save theme preference
    Crashify.utils.storage.set('theme', theme);
    
    // Emit theme change event
    Crashify.events.dispatchEvent(new CustomEvent('themeChanged', { detail: { theme } }));
}

function changeTheme(theme) {
    initializeTheme(theme);
    
    // Update theme selector UI
    Crashify.utils.$$('.theme-option').forEach(option => {
        option.classList.remove('active');
        if (option.dataset.theme === theme) {
            option.classList.add('active');
        }
    });
    
    Crashify.ui.toast.success(`Theme changed to ${theme}`);
}

// Language Management
function setLanguage(language, isRTL = false) {
    Crashify.config.language = language;
    document.documentElement.lang = language;
    document.documentElement.dir = isRTL ? 'rtl' : 'ltr';
    document.body.classList.toggle('rtl', isRTL);
    
    // Save language preference
    Crashify.utils.storage.set('language', language);
    
    // Emit language change event
    Crashify.events.dispatchEvent(new CustomEvent('languageChanged', { detail: { language, isRTL } }));
}

function changeLanguage(language) {
    // This would typically reload the page with new language
    window.location.href = `${window.location.pathname}?lang=${language}`;
}

// User Session Management
function initializeUserSession(user) {
    Crashify.state.user = user;
    Crashify.state.isLoggedIn = !!user;
    
    if (user) {
        // Initialize user-specific features
        loadUserNotifications();
        startBalanceUpdates();
    }
}

async function loadUserNotifications() {
    try {
        const notifications = await Crashify.api.getNotifications();
        Crashify.state.notifications = notifications;
        updateNotificationUI();
    } catch (error) {
        console.error('Failed to load notifications:', error);
    }
}

function updateNotificationUI() {
    const badge = Crashify.utils.$('.notification-badge');
    const unreadCount = Crashify.state.notifications.filter(n => !n.read).length;
    
    if (badge) {
        badge.textContent = unreadCount;
        badge.style.display = unreadCount > 0 ? 'block' : 'none';
    }
}

// Modal Management
function openModal(modalId) {
    const modal = Crashify.utils.$(`#${modalId}`);
    if (modal) {
        modal.classList.add('active');
        const backdrop = modal.closest('.modal-backdrop');
        if (backdrop) {
            backdrop.classList.add('active');
        }
        
        // Prevent body scroll
        document.body.style.overflow = 'hidden';
        
        // Focus management
        const firstFocusable = modal.querySelector('input, button, select, textarea, [tabindex]:not([tabindex="-1"])');
        if (firstFocusable) {
            firstFocusable.focus();
        }
        
        Crashify.state.modals.set(modalId, modal);
    }
}

function closeModal(modalId) {
    const modal = Crashify.state.modals.get(modalId) || Crashify.utils.$(`#${modalId}`);
    if (modal) {
        modal.classList.remove('active');
        const backdrop = modal.closest('.modal-backdrop');
        if (backdrop) {
            backdrop.classList.remove('active');
        }
        
        // Restore body scroll
        document.body.style.overflow = '';
        
        Crashify.state.modals.delete(modalId);
    }
}

// Dropdown Management
function toggleDropdown(dropdownId) {
    const dropdown = Crashify.utils.$(`#${dropdownId}`);
    if (dropdown) {
        const isOpen = dropdown.classList.contains('active');
        
        // Close all other dropdowns
        Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
            menu.classList.remove('active');
        });
        
        if (!isOpen) {
            dropdown.classList.add('active');
            Crashify.state.dropdowns.set(dropdownId, dropdown);
        } else {
            Crashify.state.dropdowns.delete(dropdownId);
        }
    }
}

// Mobile Menu
function toggleMobileMenu() {
    const mobileNav = Crashify.utils.$('#mobileNav');
    const menuToggle = Crashify.utils.$('.mobile-menu-toggle');
    
    if (mobileNav && menuToggle) {
        const isOpen = mobileNav.classList.contains('active');
        
        mobileNav.classList.toggle('active');
        menuToggle.classList.toggle('active');
        document.body.classList.toggle('mobile-menu-open');
    }
}

// Scroll to Top
function scrollToTop() {
    window.scrollTo({
        top: 0,
        behavior: 'smooth'
    });
}

// Back to Top Button
function initializeBackToTop() {
    const backToTopBtn = Crashify.utils.$('#backToTop');
    if (backToTopBtn) {
        const toggleVisibility = Crashify.utils.throttle(() => {
            if (window.pageYOffset > 300) {
                backToTopBtn.style.display = 'flex';
            } else {
                backToTopBtn.style.display = 'none';
            }
        }, 100);
        
        window.addEventListener('scroll', toggleVisibility);
        toggleVisibility(); // Initial check
    }
}

// Loading Screen
function hideLoadingScreen() {
    const loadingScreen = Crashify.utils.$('#loadingScreen');
    if (loadingScreen) {
        setTimeout(() => {
            Crashify.utils.fadeOut(loadingScreen, 500);
        }, 500);
    }
}

// Initialize UI Components
function initializeUI() {
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize back to top
    initializeBackToTop();
    
    // Initialize form validation
    initializeFormValidation();
    
    // Initialize keyboard shortcuts
    initializeKeyboardShortcuts();
    
    // Initialize click outside handlers
    initializeClickOutside();
}

function initializeTooltips() {
    // Simple tooltip implementation
    Crashify.utils.$$('[title]').forEach(element => {
        element.addEventListener('mouseenter', showTooltip);
        element.addEventListener('mouseleave', hideTooltip);
    });
}

function showTooltip(event) {
    const element = event.target;
    const title = element.getAttribute('title');
    
    if (title) {
        element.setAttribute('data-original-title', title);
        element.removeAttribute('title');
        
        const tooltip = Crashify.utils.createElement('div', {
            className: 'tooltip',
            innerHTML: title,
            style: 'position: absolute; z-index: 1070; background: #333; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; pointer-events: none;'
        });
        
        document.body.appendChild(tooltip);
        
        const rect = element.getBoundingClientRect();
        tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
        tooltip.style.top = rect.top - tooltip.offsetHeight - 5 + 'px';
        
        element._tooltip = tooltip;
    }
}

function hideTooltip(event) {
    const element = event.target;
    const originalTitle = element.getAttribute('data-original-title');
    
    if (originalTitle) {
        element.setAttribute('title', originalTitle);
        element.removeAttribute('data-original-title');
    }
    
    if (element._tooltip) {
        document.body.removeChild(element._tooltip);
        delete element._tooltip;
    }
}

function initializeFormValidation() {
    Crashify.utils.$$('form[data-validate]').forEach(form => {
        form.addEventListener('submit', validateForm);
    });
}

function validateForm(event) {
    const form = event.target;
    let isValid = true;
    
    // Clear previous errors
    Crashify.utils.$$(form, '.form-error').forEach(error => {
        error.remove();
    });
    
    // Validate required fields
    Crashify.utils.$$(form, '[required]').forEach(field => {
        if (!field.value.trim()) {
            showFieldError(field, 'This field is required');
            isValid = false;
        }
    });
    
    // Validate email fields
    Crashify.utils.$$(form, 'input[type="email"]').forEach(field => {
        if (field.value && !Crashify.utils.validateEmail(field.value)) {
            showFieldError(field, 'Please enter a valid email address');
            isValid = false;
        }
    });
    
    // Validate password fields
    Crashify.utils.$$(form, 'input[data-validate="password"]').forEach(field => {
        if (field.value && !Crashify.utils.validatePassword(field.value)) {
            showFieldError(field, 'Password must be at least 8 characters with uppercase, lowercase, and number');
            isValid = false;
        }
    });
    
    if (!isValid) {
        event.preventDefault();
    }
}

function showFieldError(field, message) {
    const error = Crashify.utils.createElement('div', {
        className: 'form-error',
        innerHTML: message
    });
    
    field.parentNode.appendChild(error);
    field.classList.add('error');
}

function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', (event) => {
        // Escape key closes modals and dropdowns
        if (event.key === 'Escape') {
            // Close modals
            Crashify.state.modals.forEach((modal, id) => {
                closeModal(id);
            });
            
            // Close dropdowns
            Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
                menu.classList.remove('active');
            });
            
            // Close mobile menu
            const mobileNav = Crashify.utils.$('#mobileNav');
            if (mobileNav && mobileNav.classList.contains('active')) {
                toggleMobileMenu();
            }
        }
        
        // Ctrl/Cmd + K for search
        if ((event.ctrlKey || event.metaKey) && event.key === 'k') {
            event.preventDefault();
            const searchInput = Crashify.utils.$('#searchInput');
            if (searchInput) {
                searchInput.focus();
            }
        }
    });
}

function initializeClickOutside() {
    document.addEventListener('click', (event) => {
        // Close dropdowns when clicking outside
        if (!event.target.closest('.dropdown')) {
            Crashify.utils.$$('.dropdown-menu.active').forEach(menu => {
                menu.classList.remove('active');
            });
            Crashify.state.dropdowns.clear();
        }
        
        // Close mobile menu when clicking outside
        const mobileNav = Crashify.utils.$('#mobileNav');
        if (mobileNav && mobileNav.classList.contains('active') && !event.target.closest('.mobile-nav, .mobile-menu-toggle')) {
            toggleMobileMenu();
        }
    });
}

// Cookie Consent
function checkCookieConsent() {
    const consent = Crashify.utils.storage.get('cookieConsent');
    if (!consent) {
        const banner = Crashify.utils.$('#cookieBanner');
        if (banner) {
            banner.style.display = 'block';
        }
    }
}

function acceptCookies() {
    Crashify.utils.storage.set('cookieConsent', true);
    const banner = Crashify.utils.$('#cookieBanner');
    if (banner) {
        Crashify.utils.fadeOut(banner);
    }
}

function showCookieSettings() {
    openModal('cookieSettingsModal');
}

// Export global functions
window.initializeTheme = initializeTheme;
window.changeTheme = changeTheme;
window.setLanguage = setLanguage;
window.changeLanguage = changeLanguage;
window.initializeUserSession = initializeUserSession;
window.openModal = openModal;
window.closeModal = closeModal;
window.toggleDropdown = toggleDropdown;
window.toggleMobileMenu = toggleMobileMenu;
window.scrollToTop = scrollToTop;
window.hideLoadingScreen = hideLoadingScreen;
window.initializeUI = initializeUI;
window.checkCookieConsent = checkCookieConsent;
window.acceptCookies = acceptCookies;
window.showCookieSettings = showCookieSettings;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeUI);
} else {
    initializeUI();
}

console.log('Crashify main.js loaded successfully');
