<?php
// api/sports.php - Sports API
// API ورزشی

require_once '../config/config.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'sports':
            handleSportsRequest($method);
            break;
            
        case 'leagues':
            handleLeaguesRequest($method);
            break;
            
        case 'matches':
            handleMatchesRequest($method);
            break;
            
        case 'live':
            handleLiveMatchesRequest($method);
            break;
            
        case 'bet':
            handleBetRequest($method);
            break;
            
        default:
            throw new Exception('Invalid endpoint');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handleSportsRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Mock sports data - in production, this would come from a sports data provider
    $sports = [
        [
            'id' => 1,
            'name' => 'Football',
            'slug' => 'football',
            'icon' => 'fas fa-football-ball',
            'active' => true,
            'leagues_count' => 25,
            'live_matches' => 8
        ],
        [
            'id' => 2,
            'name' => 'Basketball',
            'slug' => 'basketball',
            'icon' => 'fas fa-basketball-ball',
            'active' => true,
            'leagues_count' => 15,
            'live_matches' => 3
        ],
        [
            'id' => 3,
            'name' => 'Tennis',
            'slug' => 'tennis',
            'icon' => 'fas fa-table-tennis',
            'active' => true,
            'leagues_count' => 12,
            'live_matches' => 5
        ],
        [
            'id' => 4,
            'name' => 'Soccer',
            'slug' => 'soccer',
            'icon' => 'fas fa-futbol',
            'active' => true,
            'leagues_count' => 30,
            'live_matches' => 12
        ],
        [
            'id' => 5,
            'name' => 'Baseball',
            'slug' => 'baseball',
            'icon' => 'fas fa-baseball-ball',
            'active' => true,
            'leagues_count' => 8,
            'live_matches' => 2
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $sports
    ]);
}

function handleLeaguesRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $sportId = intval($_GET['sport_id'] ?? 0);
    
    // Mock leagues data
    $leagues = [
        [
            'id' => 1,
            'sport_id' => 1,
            'name' => 'NFL',
            'country' => 'USA',
            'logo' => '/assets/images/leagues/nfl.png',
            'active' => true,
            'matches_count' => 16,
            'live_matches' => 2
        ],
        [
            'id' => 2,
            'sport_id' => 1,
            'name' => 'Premier League',
            'country' => 'England',
            'logo' => '/assets/images/leagues/premier-league.png',
            'active' => true,
            'matches_count' => 20,
            'live_matches' => 3
        ],
        [
            'id' => 3,
            'sport_id' => 2,
            'name' => 'NBA',
            'country' => 'USA',
            'logo' => '/assets/images/leagues/nba.png',
            'active' => true,
            'matches_count' => 12,
            'live_matches' => 1
        ],
        [
            'id' => 4,
            'sport_id' => 4,
            'name' => 'Champions League',
            'country' => 'Europe',
            'logo' => '/assets/images/leagues/champions-league.png',
            'active' => true,
            'matches_count' => 8,
            'live_matches' => 2
        ]
    ];
    
    if ($sportId > 0) {
        $leagues = array_filter($leagues, function($league) use ($sportId) {
            return $league['sport_id'] === $sportId;
        });
        $leagues = array_values($leagues);
    }
    
    echo json_encode([
        'success' => true,
        'data' => $leagues
    ]);
}

function handleMatchesRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    $leagueId = intval($_GET['league_id'] ?? 0);
    $date = $_GET['date'] ?? date('Y-m-d');
    $status = $_GET['status'] ?? 'all'; // all, upcoming, live, finished
    
    // Mock matches data
    $matches = [
        [
            'id' => 1,
            'league_id' => 1,
            'league_name' => 'NFL',
            'home_team' => 'Dallas Cowboys',
            'away_team' => 'New York Giants',
            'home_logo' => '/assets/images/teams/cowboys.png',
            'away_logo' => '/assets/images/teams/giants.png',
            'start_time' => date('Y-m-d H:i:s', strtotime('+2 hours')),
            'status' => 'upcoming',
            'home_score' => null,
            'away_score' => null,
            'odds' => [
                'home_win' => 1.85,
                'away_win' => 2.10,
                'draw' => 3.20
            ]
        ],
        [
            'id' => 2,
            'league_id' => 1,
            'league_name' => 'NFL',
            'home_team' => 'Green Bay Packers',
            'away_team' => 'Chicago Bears',
            'home_logo' => '/assets/images/teams/packers.png',
            'away_logo' => '/assets/images/teams/bears.png',
            'start_time' => date('Y-m-d H:i:s', strtotime('-1 hour')),
            'status' => 'live',
            'home_score' => 14,
            'away_score' => 7,
            'minute' => 45,
            'odds' => [
                'home_win' => 1.65,
                'away_win' => 2.45,
                'draw' => 3.80
            ]
        ],
        [
            'id' => 3,
            'league_id' => 2,
            'league_name' => 'Premier League',
            'home_team' => 'Manchester United',
            'away_team' => 'Liverpool',
            'home_logo' => '/assets/images/teams/man-utd.png',
            'away_logo' => '/assets/images/teams/liverpool.png',
            'start_time' => date('Y-m-d H:i:s', strtotime('+4 hours')),
            'status' => 'upcoming',
            'home_score' => null,
            'away_score' => null,
            'odds' => [
                'home_win' => 2.20,
                'away_win' => 1.75,
                'draw' => 3.40
            ]
        ],
        [
            'id' => 4,
            'league_id' => 3,
            'league_name' => 'NBA',
            'home_team' => 'Los Angeles Lakers',
            'away_team' => 'Boston Celtics',
            'home_logo' => '/assets/images/teams/lakers.png',
            'away_logo' => '/assets/images/teams/celtics.png',
            'start_time' => date('Y-m-d H:i:s', strtotime('-3 hours')),
            'status' => 'finished',
            'home_score' => 108,
            'away_score' => 112,
            'odds' => [
                'home_win' => 1.90,
                'away_win' => 1.95
            ]
        ]
    ];
    
    // Filter by league
    if ($leagueId > 0) {
        $matches = array_filter($matches, function($match) use ($leagueId) {
            return $match['league_id'] === $leagueId;
        });
    }
    
    // Filter by status
    if ($status !== 'all') {
        $matches = array_filter($matches, function($match) use ($status) {
            return $match['status'] === $status;
        });
    }
    
    $matches = array_values($matches);
    
    echo json_encode([
        'success' => true,
        'data' => $matches,
        'total' => count($matches)
    ]);
}

function handleLiveMatchesRequest($method) {
    if ($method !== 'GET') {
        throw new Exception('Method not allowed');
    }
    
    // Mock live matches with real-time updates
    $liveMatches = [
        [
            'id' => 2,
            'league_name' => 'NFL',
            'home_team' => 'Green Bay Packers',
            'away_team' => 'Chicago Bears',
            'home_score' => 14,
            'away_score' => 7,
            'minute' => 45,
            'period' => '2nd Quarter',
            'status' => 'live',
            'odds' => [
                'home_win' => 1.65,
                'away_win' => 2.45,
                'draw' => 3.80
            ],
            'last_update' => date('Y-m-d H:i:s')
        ],
        [
            'id' => 5,
            'league_name' => 'Premier League',
            'home_team' => 'Arsenal',
            'away_team' => 'Chelsea',
            'home_score' => 1,
            'away_score' => 1,
            'minute' => 67,
            'period' => '2nd Half',
            'status' => 'live',
            'odds' => [
                'home_win' => 2.10,
                'away_win' => 2.05,
                'draw' => 3.20
            ],
            'last_update' => date('Y-m-d H:i:s')
        ],
        [
            'id' => 6,
            'league_name' => 'NBA',
            'home_team' => 'Golden State Warriors',
            'away_team' => 'Miami Heat',
            'home_score' => 89,
            'away_score' => 92,
            'minute' => 8,
            'period' => '4th Quarter',
            'status' => 'live',
            'odds' => [
                'home_win' => 1.85,
                'away_win' => 2.00
            ],
            'last_update' => date('Y-m-d H:i:s')
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $liveMatches,
        'timestamp' => time()
    ]);
}

function handleBetRequest($method) {
    if ($method !== 'POST') {
        throw new Exception('Method not allowed');
    }
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    $matchId = intval($input['match_id'] ?? 0);
    $betType = $input['bet_type'] ?? '';
    $amount = floatval($input['amount'] ?? 0);
    $odds = floatval($input['odds'] ?? 0);
    $userId = intval($input['user_id'] ?? 0);
    
    if (!$matchId || !$betType || $amount <= 0 || $odds <= 0 || !$userId) {
        throw new Exception('Missing required parameters');
    }
    
    // Validate user and balance
    $user = fetchRow("SELECT * FROM users WHERE id = ? AND status = 'active'", [$userId]);
    if (!$user) {
        throw new Exception('Invalid user');
    }
    
    if ($user['balance'] < $amount) {
        throw new Exception('Insufficient balance');
    }
    
    // Get match details
    $match = getMatchById($matchId);
    if (!$match) {
        throw new Exception('Match not found');
    }
    
    if ($match['status'] !== 'upcoming' && $match['status'] !== 'live') {
        throw new Exception('Betting not available for this match');
    }
    
    // Calculate potential payout
    $potentialPayout = $amount * $odds;
    
    // Create bet record
    $betId = execute(
        "INSERT INTO sports_bets (user_id, match_id, bet_type, amount, odds, potential_payout, status, created_at) 
         VALUES (?, ?, ?, ?, ?, ?, 'pending', NOW())",
        [$userId, $matchId, $betType, $amount, $odds, $potentialPayout]
    );
    
    // Deduct amount from user balance
    execute("UPDATE users SET balance = balance - ? WHERE id = ?", [$amount, $userId]);
    
    // Create transaction record
    execute(
        "INSERT INTO transactions (user_id, type, amount, status, description, created_at) 
         VALUES (?, 'sports_bet', ?, 'completed', ?, NOW())",
        [$userId, $amount, "Sports bet on {$match['home_team']} vs {$match['away_team']}"]
    );
    
    echo json_encode([
        'success' => true,
        'bet_id' => $betId,
        'message' => 'Bet placed successfully',
        'bet_details' => [
            'match' => $match['home_team'] . ' vs ' . $match['away_team'],
            'bet_type' => $betType,
            'amount' => $amount,
            'odds' => $odds,
            'potential_payout' => $potentialPayout
        ]
    ]);
}

function getMatchById($matchId) {
    // In a real application, this would query the database
    // For now, return mock data
    $matches = [
        1 => [
            'id' => 1,
            'home_team' => 'Dallas Cowboys',
            'away_team' => 'New York Giants',
            'status' => 'upcoming'
        ],
        2 => [
            'id' => 2,
            'home_team' => 'Green Bay Packers',
            'away_team' => 'Chicago Bears',
            'status' => 'live'
        ]
    ];
    
    return $matches[$matchId] ?? null;
}

// Helper function to get current odds for a match
function getCurrentOdds($matchId) {
    // Mock odds that would normally come from odds provider
    return [
        'home_win' => 1.85 + (rand(-20, 20) / 100),
        'away_win' => 2.10 + (rand(-20, 20) / 100),
        'draw' => 3.20 + (rand(-30, 30) / 100)
    ];
}

// Helper function to update live match scores
function updateLiveScores() {
    // This would be called by a background process to update live scores
    // Mock implementation
    return true;
}
?>

