<?php
require_once 'db.php';

class Functions {
    private $db;
    
    public function __construct() {
        $this->db = (new Database())->getConnection();
    }
    
    // تابع تولید ضریب Crash
    public function generateMultiplier() {
        // الگوریتم مشابه Crashino بر اساس توزیع احتمال
        $r = mt_rand(1, 10000) / 10000;
        $multiplier = (1 - 0.99) / (1 - $r) - 0.99;
        $multiplier = max(1, round($multiplier, 2));
        
        // اعمال advantage خانه (house edge)
        $house_edge = $this->getSetting('house_edge');
        $multiplier = $multiplier * (1 - ($house_edge / 100));
        
        return round($multiplier, 2);
    }
    
    // دریافت تنظیمات
    public function getSetting($key) {
        $query = "SELECT value FROM settings WHERE `key` = :key";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":key", $key);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch();
            return $row['value'];
        }
        return null;
    }
    
    // ذخیره تراکنش
    public function addTransaction($user_id, $type, $amount, $status = 'completed', $details = '') {
        $query = "INSERT INTO transactions (user_id, type, amount, status, details, created_at) 
                  VALUES (:user_id, :type, :amount, :status, :details, NOW())";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":type", $type);
        $stmt->bindParam(":amount", $amount);
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":details", $details);
        return $stmt->execute();
    }
    
    // شروع بازی جدید
    public function startGame() {
        $multiplier = $this->generateMultiplier();
        
        $query = "INSERT INTO games (crashed_at) VALUES (:crashed_at)";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":crashed_at", $multiplier);
        $stmt->execute();
        
        return array(
            'game_id' => $this->db->lastInsertId(),
            'multiplier' => $multiplier
        );
    }
    
    // ثبت شرط
    public function placeBet($user_id, $game_id, $amount, $auto_cashout = null) {
        // بررسی موجودی کاربر
        $user_query = "SELECT balance FROM users WHERE id = :user_id";
        $user_stmt = $this->db->prepare($user_query);
        $user_stmt->bindParam(":user_id", $user_id);
        $user_stmt->execute();
        $user = $user_stmt->fetch();
        
        if ($user['balance'] < $amount) {
            return false; // موجودی ناکافی
        }
        
        // کسر از موجودی کاربر
        $this->updateBalance($user_id, -$amount);
        
        // ثبت تراکنش
        $this->addTransaction($user_id, 'bet', $amount, 'completed', "Bet on game #$game_id");
        
        // ثبت شرط
        $query = "INSERT INTO bets (game_id, user_id, amount, auto_cashout) 
                  VALUES (:game_id, :user_id, :amount, :auto_cashout)";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":game_id", $game_id);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":amount", $amount);
        $stmt->bindParam(":auto_cashout", $auto_cashout);
        
        return $stmt->execute();
    }
    
    // نقد کردن شرط
    public function cashout($user_id, $game_id, $multiplier) {
        // دریافت شرط کاربر
        $query = "SELECT id, amount, auto_cashout FROM bets 
                  WHERE user_id = :user_id AND game_id = :game_id AND cashed_out IS NULL";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":game_id", $game_id);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $bet = $stmt->fetch();
            $win_amount = $bet['amount'] * $multiplier;
            
            // به روز رسانی شرط
            $update_query = "UPDATE bets SET cashed_out = :multiplier, profit = :profit 
                             WHERE id = :bet_id";
            $update_stmt = $this->db->prepare($update_query);
            $update_stmt->bindParam(":multiplier", $multiplier);
            $update_stmt->bindParam(":profit", $win_amount);
            $update_stmt->bindParam(":bet_id", $bet['id']);
            $update_stmt->execute();
            
            // واریز سود به کاربر
            $this->updateBalance($user_id, $win_amount);
            
            // ثبت تراکنش
            $this->addTransaction($user_id, 'win', $win_amount, 'completed', "Win from game #$game_id");
            
            return $win_amount;
        }
        return false;
    }
    
    // دریافت تاریخچه بازی‌های کاربر
    public function getUserGameHistory($user_id, $limit = 10) {
        $query = "SELECT g.id, g.crashed_at, b.amount, b.cashed_out, b.profit, g.created_at 
                  FROM games g 
                  JOIN bets b ON g.id = b.game_id 
                  WHERE b.user_id = :user_id 
                  ORDER BY g.created_at DESC 
                  LIMIT :limit";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":limit", $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    // دریافت بازی‌های اخیر
    public function getRecentGames($limit = 20) {
        $query = "SELECT g.id, g.crashed_at, COUNT(b.id) as bets, 
                  SUM(b.amount) as total_bet, MAX(b.cashed_out) as max_cashout, 
                  g.created_at 
                  FROM games g 
                  LEFT JOIN bets b ON g.id = b.game_id 
                  GROUP BY g.id 
                  ORDER BY g.created_at DESC 
                  LIMIT :limit";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":limit", $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }
    // محاسبه Cashback برای کاربر
public function calculateCashback($user_id) {
    $query = "SELECT SUM(amount) as total_loss FROM bets 
              WHERE user_id = :user_id AND profit < 0 
              AND created_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK)";
    $stmt = $this->db->prepare($query);
    $stmt->bindParam(":user_id", $user_id);
    $stmt->execute();
    $result = $stmt->fetch();
    
    $total_loss = $result['total_loss'] ?? 0;
    $cashback_percentage = $this->getVipCashbackPercentage($user_id);
    $cashback_amount = $total_loss * ($cashback_percentage / 100);
    
    return round($cashback_amount, 2);
}

// دریافت درصد Cashback بر اساس سطح VIP
public function getVipCashbackPercentage($user_id) {
    $query = "SELECT vip_level FROM users WHERE id = :user_id";
    $stmt = $this->db->prepare($query);
    $stmt->bindParam(":user_id", $user_id);
    $stmt->execute();
    $user = $stmt->fetch();
    
    $vip_level = $user['vip_level'] ?? 0;
    
    switch ($vip_level) {
        case 1: return 5;
        case 2: return 7;
        case 3: return 10;
        case 4: return 12;
        case 5: return 15;
        default: return 0;
    }
}

// ارسال Cashback به کاربر
public function sendCashback($user_id) {
    $cashback_amount = $this->calculateCashback($user_id);
    
    if ($cashback_amount > 0) {
        // افزودن Cashback به موجودی کاربر
        $this->updateBalance($user_id, $cashback_amount);
        
        // ثبت تراکنش
        $this->addTransaction($user_id, 'cashback', $cashback_amount, 'completed', "Weekly cashback");
        
        return $cashback_amount;
    }
    
    return 0;
}
    // به روز رسانی زبان کاربر
    public function updateUserLanguage($user_id, $language) {
        $query = "UPDATE users SET language = :language WHERE id = :user_id";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(":language", $language);
        $stmt->bindParam(":user_id", $user_id);
        
        if ($stmt->execute()) {
            $_SESSION['language'] = $language;
            return true;
        }
        return false;
    }
}
?>