<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Crashify - بازی Crash پیشرفته</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #ff4757;
            --secondary: #2ed573;
            --dark: #1e272e;
            --light: #f5f6fa;
            --gray: #808e9b;
            --background: #1e272e;
            --text: #f5f6fa;
            --card-bg: rgba(255, 255, 255, 0.05);
            --border: rgba(255, 255, 255, 0.1);
            --special: linear-gradient(135deg, #ff4757 0%, #ff6b81 100%);
            --rocket-color: #2ed573;
            --graph-max-scale: 20;
            --particle-color-1: #ff4757;
            --particle-color-2: #2ed573;
            --particle-color-3: #ff9f1a;
            --particle-color-4: #18dcff;
        }

        .theme-rocket {
            --primary: #ff9f1a;
            --secondary: #18dcff;
            --background: #0a3d62;
            --card-bg: rgba(255, 255, 255, 0.06);
            --special: linear-gradient(135deg, #ff9f1a 0%, #ffaf40 100%);
            --rocket-color: #ff9f1a;
            --particle-color-1: #ff9f1a;
            --particle-color-2: #18dcff;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><circle cx="20" cy="30" r="2" fill="white" opacity="0.3"/><circle cx="50" cy="10" r="1" fill="white" opacity="0.3"/><circle cx="80" cy="40" r="1.5" fill="white" opacity="0.3"/><circle cx="65" cy="70" r="0.8" fill="white" opacity="0.3"/><circle cx="30" cy="85" r="1.2" fill="white" opacity="0.3"/><circle cx="90" cy="90" r="1" fill="white" opacity="0.3"/></svg>');
        }

        .theme-space {
            --primary: #9b59b6;
            --secondary: #48dbfb;
            --background: #2c3e50;
            --card-bg: rgba(255, 255, 255, 0.07);
            --special: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
            --rocket-color: #9b59b6;
            --particle-color-1: #9b59b6;
            --particle-color-2: #48dbfb;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><circle cx="15" cy="25" r="1.2" fill="white" opacity="0.5"/><circle cx="40" cy="10" r="0.8" fill="white" opacity="0.5"/><circle cx="70" cy="35" r="1" fill="white" opacity="0.5"/><circle cx="85" cy="60" r="0.7" fill="white" opacity="0.5"/><circle cx="25" cy="80" r="1.5" fill="white" opacity="0.5"/><circle cx="60" cy="90" r="0.9" fill="white" opacity="0.5"/></svg>');
        }

        .theme-gold {
            --primary: #ffd700;
            --secondary: #ff9f1a;
            --background: #1e272e;
            --card-bg: rgba(255, 215, 0, 0.05);
            --special: linear-gradient(135deg, #ffd700 0%, #ffb142 100%);
            --rocket-color: #ffd700;
            --particle-color-1: #ffd700;
            --particle-color-2: #ff9f1a;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><path d="M20,30 Q30,20 40,30 T60,30 T80,30" stroke="rgba(255,215,0,0.1)" fill="none"/><path d="M10,60 Q20,50 30,60 T50,60 T70,60" stroke="rgba(255,215,0,0.1)" fill="none"/></svg>');
        }

        .theme-diamond {
            --primary: #00cec9;
            --secondary: #81ecec;
            --background: #0c2461;
            --card-bg: rgba(0, 206, 201, 0.05);
            --special: linear-gradient(135deg, #00cec9 0%, #00a8ff 100%);
            --rocket-color: #00cec9;
            --particle-color-1: #00cec9;
            --particle-color-2: #81ecec;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><polygon points="15,20 20,15 25,20 20,25" fill="rgba(0,206,201,0.1)"/><polygon points="40,40 45,35 50,40 45,45" fill="rgba(0,206,201,0.1)"/><polygon points="70,60 75,55 80,60 75,65" fill="rgba(0,206,201,0.1)"/><polygon points="30,80 35,75 40,80 35,85" fill="rgba(0,206,201,0.1)"/></svg>');
        }

        .theme-forest {
            --primary: #00b894;
            --secondary: #55efc4;
            --background: #004d40;
            --card-bg: rgba(0, 184, 148, 0.05);
            --special: linear-gradient(135deg, #00b894 0%, #55efc4 100%);
            --rocket-color: #00b894;
            --particle-color-1: #00b894;
            --particle-color-2: #55efc4;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><path d="M20,80 L25,70 L30,75 L35,65 L40,80" stroke="rgba(0,184,148,0.15)" fill="none"/><path d="M60,80 L65,60 L70,75 L75,55 L80,80" stroke="rgba(0,184,148,0.15)" fill="none"/></svg>');
        }

        .theme-jet {
            --primary: #dfe6e9;
            --secondary: #b2bec3;
            --background: #2d3436;
            --card-bg: rgba(223, 230, 233, 0.05);
            --special: linear-gradient(135deg, #dfe6e9 0%, #b2bec3 100%);
            --rocket-color: #dfe6e9;
            --particle-color-1: #dfe6e9;
            --particle-color-2: #b2bec3;
            --background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0 0 100 100"><path d="M10,30 L15,25 L20,30" stroke="rgba(223,230,233,0.1)" fill="none"/><path d="M30,50 L35,45 L40,50" stroke="rgba(223,230,233,0.1)" fill="none"/><path d="M60,40 L65,35 L70,40" stroke="rgba(223,230,233,0.1)" fill="none"/><path d="M80,70 L85,65 L90,70" stroke="rgba(223,230,233,0.1)" fill="none"/></svg>');
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Vazirmatn', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background-color: var(--background);
            color: var(--text);
            min-height: 100vh;
            background-image: 
                var(--background-image, radial-gradient(circle at 10% 20%, rgba(255, 255, 255, 0.03) 0%, transparent 20%)),
                radial-gradient(circle at 90% 80%, rgba(255, 255, 255, 0.03) 0%, transparent 20%);
            overflow-x: hidden;
            transition: background-color 0.5s ease;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            margin-bottom: 30px;
            border-bottom: 1px solid var(--border);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 28px;
            font-weight: bold;
            background: var(--special);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .logo i {
            font-size: 32px;
        }

        .user-panel {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .balance {
            background: var(--card-bg);
            padding: 10px 20px;
            border-radius: 12px;
            font-weight: bold;
            color: var(--secondary);
            border: 1px solid var(--border);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
            box-shadow: 0 4px 12px color-mix(in srgb, var(--primary) 30%, transparent);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px color-mix(in srgb, var(--primary) 40%, transparent);
        }

        .btn-secondary {
            background: var(--secondary);
            color: var(--dark);
            box-shadow: 0 4px 12px color-mix(in srgb, var(--secondary) 30%, transparent);
        }

        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px color-mix(in srgb, var(--secondary) 40%, transparent);
        }

        .theme-selector {
            display: flex;
            gap: 15px;
            margin: 20px 0;
            justify-content: center;
            flex-wrap: wrap;
        }

        .theme-btn {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            cursor: pointer;
            border: 2px solid transparent;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .theme-btn:hover {
            transform: scale(1.1);
        }

        .theme-btn.active {
            border-color: white;
            transform: scale(1.1);
        }

        .theme-btn .theme-tooltip {
            position: absolute;
            bottom: -30px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            opacity: 0;
            transition: opacity 0.3s;
            pointer-events: none;
            white-space: nowrap;
        }

        .theme-btn:hover .theme-tooltip {
            opacity: 1;
        }

        .theme-default { background: linear-gradient(135deg, #ff4757 0%, #ff6b81 100%); }
        .theme-rocket-btn { background: linear-gradient(135deg, #ff9f1a 0%, #ffaf40 100%); }
        .theme-space-btn { background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%); }
        .theme-gold-btn { background: linear-gradient(135deg, #ffd700 0%, #ffb142 100%); }
        .theme-diamond-btn { background: linear-gradient(135deg, #00cec9 0%, #00a8ff 100%); }
        .theme-forest-btn { background: linear-gradient(135deg, #00b894 0%, #55efc4 100%); }
        .theme-jet-btn { background: linear-gradient(135deg, #dfe6e9 0%, #b2bec3 100%); }

        .game-container {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 25px;
        }

        .game-area {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 25px;
            position: relative;
            overflow: hidden;
            border: 1px solid var(--border);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .game-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .game-timer {
            font-size: 18px;
            font-weight: bold;
            color: var(--primary);
        }

        .multiplier-display {
            font-size: 60px;
            font-weight: bold;
            text-align: center;
            margin: 30px 0;
            color: var(--secondary);
            text-shadow: 0 0 15px color-mix(in srgb, var(--secondary) 50%, transparent);
            height: 70px;
            transition: all 0.1s ease;
        }

        .graph-container {
            height: 350px;
            position: relative;
            margin: 25px 0;
            background: rgba(0, 0, 0, 0.2);
            border-radius: 15px;
            overflow: hidden;
            border: 1px solid var(--border);
        }

        canvas {
            width: 100%;
            height: 100%;
        }

        .game-controls {
            display: grid;
            grid-template-columns: 1fr auto auto;
            gap: 15px;
            margin: 25px 0;
        }

        .bet-input-container {
            position: relative;
        }

        .bet-input {
            width: 100%;
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 15px;
            color: var(--text);
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .bet-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 2px color-mix(in srgb, var(--primary) 20%, transparent);
        }

        .quick-bets {
            display: flex;
            gap: 10px;
            margin-top: 10px;
            flex-wrap: wrap;
        }

        .quick-bet-btn {
            padding: 6px 12px;
            background: var(--card-bg);
            border: 1px solid var(--border);
            border-radius: 6px;
            color: var(--text);
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .quick-bet-btn:hover {
            background: var(--primary);
            color: white;
        }

        .auto-cashout {
            display: flex;
            align-items: center;
            gap: 10px;
            margin: 15px 0;
            background: var(--card-bg);
            padding: 15px;
            border-radius: 10px;
            border: 1px solid var(--border);
            flex-wrap: wrap;
        }

        .auto-cashout input {
            width: 80px;
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid var(--border);
            border-radius: 6px;
            padding: 10px;
            color: var(--text);
            text-align: center;
        }

        .side-panel {
            display: flex;
            flex-direction: column;
            gap: 25px;
        }

        .panel {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 20px;
            border: 1px solid var(--border);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .panel-title {
            font-size: 20px;
            margin-bottom: 15px;
            color: var(--primary);
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--border);
        }

        .history-list {
            max-height: 300px;
            overflow-y: auto;
        }

        .history-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid var(--border);
        }

        .multiplier {
            font-weight: bold;
            padding: 4px 10px;
            border-radius: 6px;
        }

        .multiplier.high {
            background: color-mix(in srgb, var(--secondary) 15%, transparent);
            color: var(--secondary);
        }

        .multiplier.medium {
            background: color-mix(in srgb, #ffa502 15%, transparent);
            color: #ffa502;
        }

        .multiplier.low {
            background: color-mix(in srgb, var(--primary) 15%, transparent);
            color: var(--primary);
        }

        .bets-list {
            max-height: 300px;
            overflow-y: auto;
        }

        .bet-item {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid var(--border);
        }

        .user-bet {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: bold;
        }

        .stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
            margin: 25px 0;
        }

        .stat-box {
            background: var(--card-bg);
            padding: 20px;
            border-radius: 15px;
            text-align: center;
            border: 1px solid var(--border);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .stat-value {
            font-size: 28px;
            font-weight: bold;
            color: var(--primary);
            margin: 10px 0;
        }

        .stat-label {
            font-size: 14px;
            opacity: 0.8;
        }

        .game-state {
            text-align: center;
            margin: 20px 0;
            font-size: 20px;
            color: var(--primary);
            font-weight: bold;
        }

        .rocket-animation {
            position: absolute;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            font-size: 40px;
            opacity: 0;
            transition: all 0.5s ease;
            color: var(--rocket-color);
            z-index: 10;
            filter: drop-shadow(0 0 8px color-mix(in srgb, var(--rocket-color) 50%, transparent));
        }

        .rocket-animation.flying {
            bottom: 70%;
            opacity: 1;
        }

        .profit-calculator {
            margin: 15px 0;
            padding: 15px;
            background: var(--card-bg);
            border-radius: 10px;
            border: 1px solid var(--border);
        }

        .profit-display {
            text-align: center;
            font-size: 18px;
            margin-top: 10px;
            color: var(--secondary);
            font-weight: bold;
        }

        .language-selector {
            display: flex;
            justify-content: center;
            margin: 20px 0;
            gap: 10px;
            flex-wrap: wrap;
        }

        .language-btn {
            padding: 8px 16px;
            background: var(--card-bg);
            border: 1px solid var(--border);
            border-radius: 8px;
            color: var(--text);
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .language-btn.active {
            background: var(--primary);
            color: white;
        }

        .particle {
            position: absolute;
            border-radius: 50%;
            background: var(--particle-color-1);
            opacity: 0;
            pointer-events: none;
            z-index: 5;
        }

        @keyframes particleAnimation {
            0% {
                transform: translate(0, 0);
                opacity: 1;
            }
            100% {
                transform: translate(var(--tx), var(--ty));
                opacity: 0;
            }
        }

        @media (max-width: 1024px) {
            .game-container {
                grid-template-columns: 1fr;
            }
            
            .stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 768px) {
            .stats {
                grid-template-columns: 1fr;
            }
            
            .game-controls {
                grid-template-columns: 1fr;
            }
            
            .user-panel {
                flex-direction: column;
                gap: 10px;
            }
            
            .auto-cashout {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>
<body class="theme-default">
    <div class="container">
        <header>
            <div class="logo">
                <i class="fas fa-rocket"></i>
                <span data-translate="game_title">Crashify</span>
            </div>
            <div class="user-panel">
                <div class="balance">
                    <i class="fas fa-coins"></i>
                    $<span id="balance">100.00</span>
                </div>
                <button class="btn btn-primary" id="login-btn">
                    <i class="fas fa-user"></i>
                    <span data-translate="login_register">ورود / ثبت‌نام</span>
                </button>
            </div>
        </header>

        <div class="language-selector">
            <button class="language-btn active" data-lang="fa">فارسی</button>
            <button class="language-btn" data-lang="en">English</button>
            <button class="language-btn" data-lang="es">Español</button>
            <button class="language-btn" data-lang="fr">Français</button>
            <button class="language-btn" data-lang="de">Deutsch</button>
            <button class="language-btn" data-lang="zh">中文</button>
            <button class="language-btn" data-lang="ru">Русский</button>
            <button class="language-btn" data-lang="ar">العربية</button>
        </div>

        <div class="theme-selector">
            <div class="theme-btn theme-default active" data-theme="default">
                <span class="theme-tooltip" data-translate="theme_default">پیش فرض</span>
            </div>
            <div class="theme-btn theme-rocket-btn" data-theme="rocket">
                <span class="theme-tooltip" data-translate="theme_rocket">موشکی</span>
            </div>
            <div class="theme-btn theme-space-btn" data-theme="space">
                <span class="theme-tooltip" data-translate="theme_space">فضایی</span>
            </div>
            <div class="theme-btn theme-gold-btn" data-theme="gold">
                <span class="theme-tooltip" data-translate="theme_gold">طلایی</span>
            </div>
            <div class="theme-btn theme-diamond-btn" data-theme="diamond">
                <span class="theme-tooltip" data-translate="theme_diamond">الماس</span>
            </div>
            <div class="theme-btn theme-forest-btn" data-theme="forest">
                <span class="theme-tooltip" data-translate="theme_forest">جنگلی</span>
            </div>
            <div class="theme-btn theme-jet-btn" data-theme="jet">
                <span class="theme-tooltip" data-translate="theme_jet">جت جنگنده</span>
            </div>
        </div>

        <div class="stats">
            <div class="stat-box">
                <div class="stat-label" data-translate="online_players">بازیکنان آنلاین</div>
                <div class="stat-value" id="online-players">1,247</div>
            </div>
            <div class="stat-box">
                <div class="stat-label" data-translate="total_bet">شرط کل این دور</div>
                <div class="stat-value" id="total-bet">2,485.50</div>
                <div class="stat-label">$</div>
            </div>
            <div class="stat-box">
                <div class="stat-label" data-translate="highest_multiplier">بالاترین ضریب</div>
                <div class="stat-value" id="highest-multiplier">18.75x</div>
            </div>
        </div>

        <div class="game-container">
            <div class="game-area">
                <div class="game-header">
                    <div class="game-timer">
                        <i class="fas fa-clock"></i>
                        <span id="next-game-timer">10</span>
                        <span data-translate="seconds_until_next">ثانیه تا شروع دور بعدی</span>
                    </div>
                    <div class="game-state" id="game-state" data-translate="processing">در حال پردازش...</div>
                </div>
                
                <div class="multiplier-display" id="multiplier">1.00x</div>
                
                <div class="rocket-animation" id="rocket">
                    <i class="fas fa-rocket"></i>
                </div>
                
                <div class="graph-container">
                    <canvas id="game-graph"></canvas>
                </div>

                <div class="game-controls">
                    <div class="bet-input-container">
                        <input type="number" class="bet-input" id="bet-amount" placeholder="مبلغ شرط" value="1.00" step="0.01" min="0.10" max="1000.00">
                        <div class="quick-bets">
                            <button class="quick-bet-btn" data-value="0.10">$0.10</button>
                            <button class="quick-bet-btn" data-value="0.50">$0.50</button>
                            <button class="quick-bet-btn" data-value="1.00">$1.00</button>
                            <button class="quick-bet-btn" data-value="5.00">$5.00</button>
                            <button class="quick-bet-btn" data-value="10.00">$10.00</button>
                        </div>
                    </div>
                    <button class="btn btn-primary" id="place-bet">
                        <i class="fas fa-play"></i>
                        <span data-translate="start_game">شروع بازی</span>
                    </button>
                    <button class="btn btn-secondary" id="cash-out" disabled>
                        <i class="fas fa-hand-holding-usd"></i>
                        <span data-translate="cash_out">نقد کردن</span>
                    </button>
                </div>

                <div class="auto-cashout">
                    <span data-translate="auto_cashout_at">نقد کردن خودکار در:</span>
                    <input type="number" id="auto-cashout" value="2.00" step="0.1" min="1.1" max="100">
                    <span>X</span>
                    <button class="btn btn-primary" id="set-auto-cashout">
                        <i class="fas fa-check"></i>
                        <span data-translate="activate">فعال‌سازی</span>
                    </button>
                </div>

                <div class="profit-calculator">
                    <div data-translate="potential_profit">سود احتمالی:</div>
                    <span id="profit-multiplier">2.00x</span>
                    <div class="profit-display" id="profit-amount">$2.00</div>
                </div>
            </div>

            <div class="side-panel">
                <div class="panel">
                    <div class="panel-title">
                        <span data-translate="game_history">تاریخچه بازی‌ها</span>
                        <i class="fas fa-history"></i>
                    </div>
                    <div class="history-list" id="history-list">
                        <div class="history-item">
                            <span class="time" data-translate="seconds_ago" data-value="2">۲ ثانیه پیش</span>
                            <span class="multiplier high">۸.۴۵x</span>
                        </div>
                        <div class="history-item">
                            <span class="time" data-translate="seconds_ago" data-value="15">۱۵ ثانیه پیش</span>
                            <span class="multiplier low">۱.۲۳x</span>
                        </div>
                        <div class="history-item">
                            <span class="time" data-translate="seconds_ago" data-value="45">۴۵ ثانیه پیش</span>
                            <span class="multiplier medium">۳.۷۶x</span>
                        </div>
                    </div>
                </div>

                <div class="panel">
                    <div class="panel-title">
                        <span data-translate="recent_bets">شرط‌های اخیر</span>
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="bets-list" id="bets-list">
                        <div class="bet-item">
                            <div class="user-bet">
                                <div class="avatar">U1</div>
                                <span>User1</span>
                            </div>
                            <div class="bet-amount">$1.00 (۲.۳۴x)</div>
                        </div>
                        <div class="bet-item">
                            <div class="user-bet">
                                <div class="avatar">U2</div>
                                <span>User2</span>
                            </div>
                            <div class="bet-amount">$0.50 (۱.۸۹x)</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // سیستم زبان‌ها
            const translations = {
                fa: {
                    game_title: "crashify",
                    login_register: "ورود / ثبت‌نام",
                    theme_default: "پیش فرض",
                    theme_rocket: "موشکی",
                    theme_space: "فضایی",
                    theme_gold: "طلایی",
                    theme_diamond: "الماس",
                    theme_forest: "جنگلی",
                    theme_jet: "جت جنگنده",
                    online_players: "بازیکنان آنلاین",
                    total_bet: "شرط کل این دور",
                    highest_multiplier: "بالاترین ضریب",
                    seconds_until_next: "ثانیه تا شروع دور بعدی",
                    processing: "در حال پردازش...",
                    start_game: "شروع بازی",
                    cash_out: "نقد کردن",
                    auto_cashout_at: "نقد کردن خودکار در:",
                    activate: "فعال‌سازی",
                    potential_profit: "سود احتمالی:",
                    game_history: "تاریخچه بازی‌ها",
                    recent_bets: "شرط‌های اخیر",
                    seconds_ago: "ثانیه پیش",
                    minutes_ago: "دقیقه پیش",
                    now: "هم اکنون"
                },
                en: {
                    game_title: "crashify",
                    login_register: "Login / Register",
                    theme_default: "Default",
                    theme_rocket: "Rocket",
                    theme_space: "Space",
                    theme_gold: "Gold",
                    theme_diamond: "Diamond",
                    theme_forest: "Forest",
                    theme_jet: "Jet Fighter",
                    online_players: "Online Players",
                    total_bet: "Total Bet This Round",
                    highest_multiplier: "Highest Multiplier",
                    seconds_until_next: "seconds until next round",
                    processing: "Processing...",
                    start_game: "Start Game",
                    cash_out: "Cash Out",
                    auto_cashout_at: "Auto cashout at:",
                    activate: "Activate",
                    potential_profit: "Potential Profit:",
                    game_history: "Game History",
                    recent_bets: "Recent Bets",
                    seconds_ago: "seconds ago",
                    minutes_ago: "minutes ago",
                    now: "Just now"
                },
                es: {
                    game_title: "crashify",
                    login_register: "Iniciar/Registrar",
                    theme_default: "Predeterminado",
                    theme_rocket: "Cohete",
                    theme_space: "Espacial",
                    theme_gold: "Oro",
                    theme_diamond: "Diamante",
                    theme_forest: "Forestal",
                    theme_jet: "Caza Jet",
                    online_players: "Jugadores en línea",
                    total_bet: "Apuesta total",
                    highest_multiplier: "Multiplicador más alto",
                    seconds_until_next: "segundos para la siguiente ronda",
                    processing: "Procesando...",
                    start_game: "Comenzar juego",
                    cash_out: "Cobrar",
                    auto_cashout_at: "Auto cobrar en:",
                    activate: "Activar",
                    potential_profit: "Ganancia potencial:",
                    game_history: "Historial de juego",
                    recent_bets: "Apuestas recientes",
                    seconds_ago: "segundos atrás",
                    minutes_ago: "minutos atrás",
                    now: "Ahora mismo"
                },
                fr: {
                    game_title: "crashify",
                    login_register: "Connexion/Inscription",
                    theme_default: "Défaut",
                    theme_rocket: "Fusée",
                    theme_space: "Spatial",
                    theme_gold: "Or",
                    theme_diamond: "Diamant",
                    theme_forest: "Forêt",
                    theme_jet: "Avion de chasse",
                    online_players: "Joueurs en ligne",
                    total_bet: "Mise totale",
                    highest_multiplier: "Multiplicateur le plus élevé",
                    seconds_until_next: "secondes jusqu'au prochain tour",
                    processing: "Traitement...",
                    start_game: "Commencer le jeu",
                    cash_out: "Encaisser",
                    auto_cashout_at: "Encaissement auto à:",
                    activate: "Activer",
                    potential_profit: "Bénéfice potentiel:",
                    game_history: "Historique du jeu",
                    recent_bets: "Paris récents",
                    seconds_ago: "secondes",
                    minutes_ago: "minutes",
                    now: "À l'instant"
                },
                de: {
                    game_title: "crashify",
                    login_register: "Anmelden/Registrieren",
                    theme_default: "Standard",
                    theme_rocket: "Rakete",
                    theme_space: "Weltraum",
                    theme_gold: "Gold",
                    theme_diamond: "Diamant",
                    theme_forest: "Wald",
                    theme_jet: "Kampfjet",
                    online_players: "Spieler online",
                    total_bet: "Gesamteinsatz",
                    highest_multiplier: "Höchster Multiplikator",
                    seconds_until_next: "Sekunden bis zur nächsten Runde",
                    processing: "Wird verarbeitet...",
                    start_game: "Spiel starten",
                    cash_out: "Auszahlen",
                    auto_cashout_at: "Auto-Auszahlung bei:",
                    activate: "Aktivieren",
                    potential_profit: "Möglicher Gewinn:",
                    game_history: "Spielverlauf",
                    recent_bets: "Letzte Wetten",
                    seconds_ago: "Sekunden ago",
                    minutes_ago: "Minuten ago",
                    now: "Gerade jetzt"
                },
                zh: {
                    game_title: "crashify",
                    login_register: "登录/注册",
                    theme_default: "默认",
                    theme_rocket: "火箭",
                    theme_space: "太空",
                    theme_gold: "黄金",
                    theme_diamond: "钻石",
                    theme_forest: "森林",
                    theme_jet: "喷气战斗机",
                    online_players: "在线玩家",
                    total_bet: "总投注额",
                    highest_multiplier: "最高倍数",
                    seconds_until_next: "秒后下一轮",
                    processing: "处理中...",
                    start_game: "开始游戏",
                    cash_out: "兑现",
                    auto_cashout_at: "自动兑现于:",
                    activate: "激活",
                    potential_profit: "潜在利润:",
                    game_history: "游戏历史",
                    recent_bets: "最近投注",
                    seconds_ago: "秒前",
                    minutes_ago: "分钟前",
                    now: "刚刚"
                },
                ru: {
                    game_title: "crashify",
                    login_register: "Вход/Регистрация",
                    theme_default: "По умолчанию",
                    theme_rocket: "Ракета",
                    theme_space: "Космос",
                    theme_gold: "Золото",
                    theme_diamond: "Алмаз",
                    theme_forest: "Лес",
                    theme_jet: "Истребитель",
                    online_players: "Игроки онлайн",
                    total_bet: "Общая ставка",
                    highest_multiplier: "Наибольший множитель",
                    seconds_until_next: "секунд до следующего раунда",
                    processing: "Обработка...",
                    start_game: "Начать игру",
                    cash_out: "Вывести",
                    auto_cashout_at: "Автовывод при:",
                    activate: "Активировать",
                    potential_profit: "Потенциальная прибыль:",
                    game_history: "История игр",
                    recent_bets: "Последние ставки",
                    seconds_ago: "секунд назад",
                    minutes_ago: "минут назад",
                    now: "Только что"
                },
                ar: {
                    game_title: "crashify",
                    login_register: "تسجيل الدخول/التسجيل",
                    theme_default: "افتراضي",
                    theme_rocket: "صاروخ",
                    theme_space: "فضاء",
                    theme_gold: "ذهب",
                    theme_diamond: "الماس",
                    theme_forest: "غابة",
                    theme_jet: "طائرة مقاتلة",
                    online_players: "اللاعبون المتصلون",
                    total_bet: "إجمالي الرهان",
                    highest_multiplier: "أعلى مضاعف",
                    seconds_until_next: "ثانية حتى الجولة التالية",
                    processing: "جاري المعالجة...",
                    start_game: "بدء اللعبة",
                    cash_out: "صرف",
                    auto_cashout_at: "صرف تلقائي عند:",
                    activate: "تفعيل",
                    potential_profit: "الربح المحتمل:",
                    game_history: "سجل اللعبة",
                    recent_bets: "الرهانات الحديثة",
                    seconds_ago: "ثانية مضت",
                    minutes_ago: "دقيقة مضت",
                    now: "الآن"
                }
            };

            let currentLanguage = 'fa';
            
            // تغییر زبان
            const languageButtons = document.querySelectorAll('.language-btn');
            languageButtons.forEach(btn => {
                btn.addEventListener('click', function() {
                    const lang = this.getAttribute('data-lang');
                    changeLanguage(lang);
                    
                    languageButtons.forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                });
            });

            function changeLanguage(lang) {
                currentLanguage = lang;
                document.documentElement.lang = lang;
                document.documentElement.dir = ['ar', 'fa'].includes(lang) ? 'rtl' : 'ltr';
                
                // به روزرسانی تمامی متون
                document.querySelectorAll('[data-translate]').forEach(element => {
                    const key = element.getAttribute('data-translate');
                    
                    if (translations[lang][key]) {
                        if (element.hasAttribute('data-value')) {
                            const value = element.getAttribute('data-value');
                            element.textContent = value + ' ' + translations[lang][key];
                        } else {
                            element.textContent = translations[lang][key];
                        }
                    }
                });
            }

            // تنظیم تم‌ها
            const themeButtons = document.querySelectorAll('.theme-btn');
            themeButtons.forEach(btn => {
                btn.addEventListener('click', function() {
                    const theme = this.getAttribute('data-theme');
                    document.body.className = 'theme-' + theme;
                    
                    themeButtons.forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                });
            });

            // متغیرهای بازی
            let balance = 100.00;
            let currentBet = 0;
            let currentMultiplier = 1.0;
            let gameRunning = false;
            let cashoutAt = 2.0;
            let autoCashoutEnabled = false;
            let gameInterval;
            let crashPoint = 0;
            let gameHistory = [];
            let nextGameTimer = 10;
            let graphMaxScale = 20;

            // عناصر DOM
            const balanceEl = document.getElementById('balance');
            const multiplierEl = document.getElementById('multiplier');
            const betAmountEl = document.getElementById('bet-amount');
            const placeBetBtn = document.getElementById('place-bet');
            const cashOutBtn = document.getElementById('cash-out');
            const autoCashoutEl = document.getElementById('auto-cashout');
            const setAutoCashoutBtn = document.getElementById('set-auto-cashout');
            const gameStateEl = document.getElementById('game-state');
            const nextGameTimerEl = document.getElementById('next-game-timer');
            const rocketEl = document.getElementById('rocket');
            const canvas = document.getElementById('game-graph');
            const ctx = canvas.getContext('2d');
            const profitMultiplierEl = document.getElementById('profit-multiplier');
            const profitAmountEl = document.getElementById('profit-amount');
            const onlinePlayersEl = document.getElementById('online-players');
            const totalBetEl = document.getElementById('total-bet');
            const highestMultiplierEl = document.getElementById('highest-multiplier');

            // ایجاد ذرات بصری
            function createParticles(x, y, count) {
                for (let i = 0; i < count; i++) {
                    const particle = document.createElement('div');
                    particle.className = 'particle';
                    
                    const size = Math.random() * 5 + 2;
                    const tx = (Math.random() - 0.5) * 100;
                    const ty = (Math.random() - 0.5) * 100;
                    
                    particle.style.width = `${size}px`;
                    particle.style.height = `${size}px`;
                    particle.style.left = `${x}px`;
                    particle.style.top = `${y}px`;
                    
                    // انتخاب رنگ تصادفی از پالت تم
                    const colors = [
                        'var(--particle-color-1)',
                        'var(--particle-color-2)',
                        'var(--particle-color-3)',
                        'var(--particle-color-4)'
                    ];
                    particle.style.background = colors[Math.floor(Math.random() * colors.length)];
                    
                    particle.style.setProperty('--tx', `${tx}px`);
                    particle.style.setProperty('--ty', `${ty}px`);
                    
                    document.querySelector('.game-area').appendChild(particle);
                    
                    particle.style.animation = `particleAnimation ${Math.random() * 1 + 0.5}s forwards`;
                    
                    setTimeout(() => {
                        particle.remove();
                    }, 700);
                }
            }

            // محاسبه نقطه Crash با الگوریتمی که به نفع خانه است
            function calculateCrashPoint() {
                const houseEdge = 0.01;
                const e = 1 - houseEdge;
                const random = Math.random();
                
                return Math.max(1, (e / (1 - random)) * 0.97);
            }

            // اندازه‌گیری Canvas
            function resizeCanvas() {
                canvas.width = canvas.offsetWidth;
                canvas.height = canvas.offsetHeight;
                drawGraph();
            }

            // رسم نمودار
            function drawGraph() {
                ctx.clearRect(0, 0, canvas.width, canvas.height);
                
                // رسم شبکه
                ctx.strokeStyle = 'rgba(255, 255, 255, 0.1)';
                ctx.lineWidth = 1;
                
                // خطوط عمودی
                for (let x = 0; x < canvas.width; x += 50) {
                    ctx.beginPath();
                    ctx.moveTo(x, 0);
                    ctx.lineTo(x, canvas.height);
                    ctx.stroke();
                }
                
                // خطوط افقی - با توجه به مقیاس پویا
                const scaleStep = graphMaxScale / 10;
                for (let i = 0; i <= 10; i++) {
                    const value = i * scaleStep;
                    const y = canvas.height - 50 - (value * (canvas.height - 100) / graphMaxScale);
                    
                    ctx.beginPath();
                    ctx.moveTo(0, y);
                    ctx.lineTo(canvas.width, y);
                    ctx.stroke();
                    
                    // نمایش مقادیر روی محور Y
                    ctx.fillStyle = 'rgba(255, 255, 255, 0.7)';
                    ctx.font = '12px Arial';
                    ctx.fillText(value.toFixed(1) + 'x', 10, y - 5);
                }
                
                if (gameRunning) {
                    // محاسبه موقعیت موشک روی نمودار
                    const progress = Math.min(currentMultiplier / crashPoint, 1);
                    const x = progress * (canvas.width - 100);
                    const y = canvas.height - 50 - (currentMultiplier * (canvas.height - 100) / graphMaxScale);
                    
                    // رسم خط نمودار
                    ctx.beginPath();
                    ctx.moveTo(0, canvas.height - 50);
                    
                    // منحنی صاف برای نمودار
                    for (let i = 1; i <= x; i++) {
                        const progressPoint = i / (canvas.width - 100);
                        const pointMultiplier = progressPoint * crashPoint;
                        const pointY = canvas.height - 50 - (pointMultiplier * (canvas.height - 100) / graphMaxScale);
                        
                        // اگر از نقطه فعلی فراتر رفتیم، خط را قطع می‌کنیم
                        if (pointMultiplier > currentMultiplier) break;
                        
                        ctx.lineTo(i, pointY);
                    }
                    
                    ctx.strokeStyle = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                    ctx.lineWidth = 3;
                    ctx.stroke();
                    
                    // نقطه روی نمودار (نوک موشک)
                    ctx.beginPath();
                    ctx.arc(x, y, 6, 0, Math.PI * 2);
                    ctx.fillStyle = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                    ctx.fill();

                    // موقعیت‌یابی موشک
                    rocketEl.style.left = `${x + 25}px`;
                    rocketEl.style.bottom = `${canvas.height - y}px`;
                    
                    // نمایش نقطه Crash
                    if (crashPoint > 0) {
                        const crashY = canvas.height - 50 - (crashPoint * (canvas.height - 100) / graphMaxScale);
                        ctx.setLineDash([5, 3]);
                        ctx.beginPath();
                        ctx.moveTo(0, crashY);
                        ctx.lineTo(canvas.width, crashY);
                        ctx.strokeStyle = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                        ctx.lineWidth = 2;
                        ctx.stroke();
                        ctx.setLineDash([]);

                        ctx.fillStyle = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                        ctx.fillText(`Crash: ${crashPoint.toFixed(2)}x`, canvas.width - 100, crashY - 10);
                    }
                }
            }

            // به‌روزرسانی ماشین حساب سود
            function updateProfitCalculator() {
                const betAmount = parseFloat(betAmountEl.value) || 0;
                const multiplier = parseFloat(autoCashoutEl.value) || 2.0;
                const profit = (betAmount * multiplier).toFixed(2);
                
                profitMultiplierEl.textContent = multiplier.toFixed(2) + 'x';
                profitAmountEl.textContent = '$' + profit;
            }

            // شروع بازی
            function startGame() {
                if (currentBet <= 0) return;
                
                crashPoint = calculateCrashPoint();
                gameRunning = true;
                currentMultiplier = 1.0;
                placeBetBtn.disabled = true;
                cashOutBtn.disabled = false;
                gameStateEl.textContent = translations[currentLanguage].processing;
                rocketEl.classList.add('flying');
                
                // تنظیم مقیاس نمودار بر اساس نقطه Crash
                graphMaxScale = Math.max(20, crashPoint * 1.2);
                
                // کاهش موجودی با میزان شرط
                balance -= currentBet;
                balanceEl.textContent = balance.toFixed(2);
                
                // شبیه‌سازی افزایش ضریب
                gameInterval = setInterval(() => {
                    // افزایش ضریب
                    const increment = 0.05 + (Math.random() * 0.1);
                    currentMultiplier += increment;
                    
                    multiplierEl.textContent = currentMultiplier.toFixed(2) + "x";
                    
                    // ایجاد ذرات بصری در اطراف موشک
                    if (Math.random() > 0.7) {
                        const rocketRect = rocketEl.getBoundingClientRect();
                        const graphRect = canvas.getBoundingClientRect();
                        
                        const x = rocketRect.left - graphRect.left + rocketRect.width / 2;
                        const y = rocketRect.top - graphRect.top + rocketRect.height / 2;
                        
                        createParticles(x, y, 3);
                    }
                    
                    // بررسی نقد کردن خودکار
                    if (autoCashoutEnabled && currentMultiplier >= cashoutAt) {
                        cashOut();
                        return;
                    }
                    
                    // بررسی crash
                    if (currentMultiplier >= crashPoint) {
                        endGame();
                        return;
                    }
                    
                    drawGraph();
                }, 100);
            }

            // نقد کردن
            function cashOut() {
                if (!gameRunning) return;
                
                const winAmount = parseFloat((currentBet * currentMultiplier).toFixed(2));
                balance += winAmount;
                balanceEl.textContent = balance.toFixed(2);
                
                endGame();
                
                // نمایش پیام سود
                gameStateEl.textContent = `${translations[currentLanguage].cash_out} ${winAmount.toFixed(2)} $`;
                gameStateEl.style.color = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                
                setTimeout(() => {
                    gameStateEl.textContent = translations[currentLanguage].processing;
                    gameStateEl.style.color = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                }, 3000);
            }

            // پایان بازی
            function endGame() {
                clearInterval(gameInterval);
                gameRunning = false;
                placeBetBtn.disabled = false;
                cashOutBtn.disabled = true;
                rocketEl.classList.remove('flying');
                
                // افزودن به تاریخچه
                addToHistory(currentMultiplier);
                
                // بازنشانی
                currentBet = 0;
                betAmountEl.value = "1.00";
                
                drawGraph();
                
                // شروع تایمر برای دور بعدی
                nextGameTimer = 10;
                const timerInterval = setInterval(() => {
                    nextGameTimer--;
                    nextGameTimerEl.textContent = nextGameTimer;
                    
                    if (nextGameTimer <= 0) {
                        clearInterval(timerInterval);
                        gameStateEl.textContent = translations[currentLanguage].start_game;
                    }
                }, 1000);
            }

            // افزودن به تاریخچه
            function addToHistory(multiplier) {
                const historyList = document.getElementById('history-list');
                const historyItem = document.createElement('div');
                historyItem.className = 'history-item';
                
                const timeSpan = document.createElement('span');
                timeSpan.className = 'time';
                timeSpan.textContent = translations[currentLanguage].now;
                
                const multiplierSpan = document.createElement('span');
                multiplierSpan.className = 'multiplier';
                
                if (multiplier >= 5) multiplierSpan.classList.add('high');
                else if (multiplier >= 2) multiplierSpan.classList.add('medium');
                else multiplierSpan.classList.add('low');
                
                multiplierSpan.textContent = multiplier.toFixed(2) + 'x';
                
                historyItem.appendChild(timeSpan);
                historyItem.appendChild(multiplierSpan);
                
                historyList.insertBefore(historyItem, historyList.firstChild);
                
                // حذف آیتم‌های اضافی
                if (historyList.children.length > 10) {
                    historyList.removeChild(historyList.lastChild);
                }
                
                // به‌روزرسانی بالاترین ضریب
                const currentHighest = parseFloat(highestMultiplierEl.textContent);
                if (multiplier > currentHighest) {
                    highestMultiplierEl.textContent = multiplier.toFixed(2) + 'x';
                }
                
                // ذخیره در تاریخچه بازی
                gameHistory.push(multiplier);
            }

            // رویدادهای کلیک
            placeBetBtn.addEventListener('click', function() {
                const betAmount = parseFloat(betAmountEl.value);
                
                if (isNaN(betAmount) || betAmount < 0.10) {
                    alert("لطفا مبلغ شرط معتبر وارد کنید (حداقل 0.10)");
                    return;
                }
                
                if (betAmount > balance) {
                    alert("موجودی شما کافی نیست!");
                    return;
                }
                
                currentBet = betAmount;
                startGame();
            });

            cashOutBtn.addEventListener('click', cashOut);

            setAutoCashoutBtn.addEventListener('click', function() {
                cashoutAt = parseFloat(autoCashoutEl.value) || 2.0;
                autoCashoutEnabled = true;
                alert(`${translations[currentLanguage].auto_cashout_at} ${cashoutAt.toFixed(2)}x ${translations[currentLanguage].activate}`);
            });

            // دکمه‌های شرط سریع
            document.querySelectorAll('.quick-bet-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const value = parseFloat(this.getAttribute('data-value'));
                    betAmountEl.value = value.toFixed(2);
                    updateProfitCalculator();
                });
            });

            // تغییرات در inputها
            betAmountEl.addEventListener('input', updateProfitCalculator);
            autoCashoutEl.addEventListener('input', updateProfitCalculator);

            // تنظیم خودکار اندازه Canvas
            window.addEventListener('resize', resizeCanvas);
            resizeCanvas();
            drawGraph();
            updateProfitCalculator();

            // شبیه‌سازی بازیکنان آنلاین و شرط‌ها
            setInterval(() => {
                // بازیکنان آنلاین
                const onlinePlayers = parseInt(onlinePlayersEl.textContent.replace(/,/g, ''));
                const change = Math.floor(Math.random() * 10) - 3;
                let newOnline = Math.max(1000, onlinePlayers + change);
                onlinePlayersEl.textContent = newOnline.toLocaleString();
                
                // شرط کل
                const totalBet = parseFloat(totalBetEl.textContent.replace(/,/g, ''));
                const betChange = parseFloat((Math.random() * 50 - 10).toFixed(2));
                let newTotalBet = Math.max(100, totalBet + betChange);
                totalBetEl.textContent = newTotalBet.toFixed(2);
            }, 5000);
        });
    </script>
</body>
</html>