<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Crash Ultimate - بازی Crash پیشرفته</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #ff4757;
            --secondary: #2ed573;
            --dark: #1e272e;
            --light: #f5f6fa;
            --gray: #808e9b;
            --background: #1e272e;
            --text: #f5f6fa;
            --card-bg: rgba(255, 255, 255, 0.05);
            --border: rgba(255, 255, 255, 0.1);
        }

        .theme-rocket {
            --primary: #ff9f1a;
            --secondary: #18dcff;
            --background: #0a3d62;
            --card-bg: rgba(255, 255, 255, 0.06);
            --special: linear-gradient(135deg, #ff9f1a 0%, #ffaf40 100%);
        }

        .theme-space {
            --primary: #9b59b6;
            --secondary: #48dbfb;
            --background: #2c3e50;
            --card-bg: rgba(255, 255, 255, 0.07);
            --special: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
        }

        .theme-gold {
            --primary: #ffd700;
            --secondary: #ff9f1a;
            --background: #1e272e;
            --card-bg: rgba(255, 215, 0, 0.05);
            --special: linear-gradient(135deg, #ffd700 0%, #ffb142 100%);
        }

        .theme-diamond {
            --primary: #00cec9;
            --secondary: #81ecec;
            --background: #0c2461;
            --card-bg: rgba(0, 206, 201, 0.05);
            --special: linear-gradient(135deg, #00cec9 0%, #00a8ff 100%);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Vazirmatn', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background-color: var(--background);
            color: var(--text);
            min-height: 100vh;
            background-image: 
                radial-gradient(circle at 10% 20%, rgba(255, 255, 255, 0.03) 0%, transparent 20%),
                radial-gradient(circle at 90% 80%, rgba(255, 255, 255, 0.03) 0%, transparent 20%);
            overflow-x: hidden;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            margin-bottom: 30px;
            border-bottom: 1px solid var(--border);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 28px;
            font-weight: bold;
            background: var(--special, linear-gradient(135deg, #ff4757 0%, #ff6b81 100%));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .logo i {
            font-size: 32px;
        }

        .user-panel {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .balance {
            background: var(--card-bg);
            padding: 10px 20px;
            border-radius: 12px;
            font-weight: bold;
            color: var(--secondary);
            border: 1px solid var(--border);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
            box-shadow: 0 4px 12px rgba(255, 71, 87, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(255, 71, 87, 0.4);
        }

        .btn-secondary {
            background: var(--secondary);
            color: var(--dark);
            box-shadow: 0 4px 12px rgba(46, 213, 115, 0.3);
        }

        .btn-secondary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(46, 213, 115, 0.4);
        }

        .theme-selector {
            display: flex;
            gap: 15px;
            margin: 20px 0;
            justify-content: center;
        }

        .theme-btn {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            cursor: pointer;
            border: 2px solid transparent;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
            transition: all 0.3s ease;
        }

        .theme-btn:hover {
            transform: scale(1.1);
        }

        .theme-btn.active {
            border-color: white;
            transform: scale(1.1);
        }

        .theme-default { background: linear-gradient(135deg, #ff4757 0%, #ff6b81 100%); }
        .theme-rocket-btn { background: linear-gradient(135deg, #ff9f1a 0%, #ffaf40 100%); }
        .theme-space-btn { background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%); }
        .theme-gold-btn { background: linear-gradient(135deg, #ffd700 0%, #ffb142 100%); }
        .theme-diamond-btn { background: linear-gradient(135deg, #00cec9 0%, #00a8ff 100%); }

        .game-container {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 25px;
        }

        .game-area {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 25px;
            position: relative;
            overflow: hidden;
            border: 1px solid var(--border);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .game-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .game-timer {
            font-size: 18px;
            font-weight: bold;
            color: var(--primary);
        }

        .multiplier-display {
            font-size: 60px;
            font-weight: bold;
            text-align: center;
            margin: 30px 0;
            color: var(--secondary);
            text-shadow: 0 0 15px rgba(46, 213, 115, 0.5);
            height: 70px;
            transition: all 0.1s ease;
        }

        .graph-container {
            height: 350px;
            position: relative;
            margin: 25px 0;
            background: rgba(0, 0, 0, 0.2);
            border-radius: 15px;
            overflow: hidden;
            border: 1px solid var(--border);
        }

        canvas {
            width: 100%;
            height: 100%;
        }

        .game-controls {
            display: grid;
            grid-template-columns: 1fr auto auto;
            gap: 15px;
            margin: 25px 0;
        }

        .bet-input-container {
            position: relative;
        }

        .bet-input {
            width: 100%;
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 15px;
            color: var(--text);
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .bet-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 2px rgba(255, 71, 87, 0.2);
        }

        .quick-bets {
            display: flex;
            gap: 10px;
            margin-top: 10px;
        }

        .quick-bet-btn {
            padding: 6px 12px;
            background: var(--card-bg);
            border: 1px solid var(--border);
            border-radius: 6px;
            color: var(--text);
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .quick-bet-btn:hover {
            background: var(--primary);
            color: white;
        }

        .auto-cashout {
            display: flex;
            align-items: center;
            gap: 10px;
            margin: 15px 0;
            background: var(--card-bg);
            padding: 15px;
            border-radius: 10px;
            border: 1px solid var(--border);
        }

        .auto-cashout input {
            width: 80px;
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid var(--border);
            border-radius: 6px;
            padding: 10px;
            color: var(--text);
            text-align: center;
        }

        .side-panel {
            display: flex;
            flex-direction: column;
            gap: 25px;
        }

        .panel {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 20px;
            border: 1px solid var(--border);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .panel-title {
            font-size: 20px;
            margin-bottom: 15px;
            color: var(--primary);
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--border);
        }

        .history-list {
            max-height: 300px;
            overflow-y: auto;
        }

        .history-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid var(--border);
        }

        .multiplier {
            font-weight: bold;
            padding: 4px 10px;
            border-radius: 6px;
        }

        .multiplier.high {
            background: rgba(46, 213, 115, 0.15);
            color: var(--secondary);
        }

        .multiplier.medium {
            background: rgba(255, 165, 2, 0.15);
            color: #ffa502;
        }

        .multiplier.low {
            background: rgba(255, 71, 87, 0.15);
            color: var(--primary);
        }

        .bets-list {
            max-height: 300px;
            overflow-y: auto;
        }

        .bet-item {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid var(--border);
        }

        .user-bet {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: bold;
        }

        .stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
            margin: 25px 0;
        }

        .stat-box {
            background: var(--card-bg);
            padding: 20px;
            border-radius: 15px;
            text-align: center;
            border: 1px solid var(--border);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .stat-value {
            font-size: 28px;
            font-weight: bold;
            color: var(--primary);
            margin: 10px 0;
        }

        .stat-label {
            font-size: 14px;
            opacity: 0.8;
        }

        .game-state {
            text-align: center;
            margin: 20px 0;
            font-size: 20px;
            color: var(--primary);
            font-weight: bold;
        }

        .rocket-animation {
            position: absolute;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            font-size: 40px;
            opacity: 0;
            transition: all 0.5s ease;
        }

        .rocket-animation.flying {
            bottom: 70%;
            opacity: 1;
        }

        .profit-calculator {
            margin: 15px 0;
            padding: 15px;
            background: var(--card-bg);
            border-radius: 10px;
            border: 1px solid var(--border);
        }

        .profit-display {
            text-align: center;
            font-size: 18px;
            margin-top: 10px;
            color: var(--secondary);
            font-weight: bold;
        }

        @media (max-width: 1024px) {
            .game-container {
                grid-template-columns: 1fr;
            }
            
            .stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 768px) {
            .stats {
                grid-template-columns: 1fr;
            }
            
            .game-controls {
                grid-template-columns: 1fr;
            }
            
            .user-panel {
                flex-direction: column;
                gap: 10px;
            }
        }
    </style>
</head>
<body class="theme-default">
    <div class="container">
        <header>
            <div class="logo">
                <i class="fas fa-rocket"></i>
                <span>Crash Ultimate</span>
            </div>
            <div class="user-panel">
                <div class="balance">
                    <i class="fas fa-coins"></i>
                    <span id="balance">5,000</span>
                    <span>سکه</span>
                </div>
                <button class="btn btn-primary" id="login-btn">
                    <i class="fas fa-user"></i>
                    <span>ورود / ثبت‌نام</span>
                </button>
            </div>
        </header>

        <div class="theme-selector">
            <div class="theme-btn theme-default active" data-theme="default"></div>
            <div class="theme-btn theme-rocket-btn" data-theme="rocket"></div>
            <div class="theme-btn theme-space-btn" data-theme="space"></div>
            <div class="theme-btn theme-gold-btn" data-theme="gold"></div>
            <div class="theme-btn theme-diamond-btn" data-theme="diamond"></div>
        </div>

        <div class="stats">
            <div class="stat-box">
                <div class="stat-label">بازیکنان آنلاین</div>
                <div class="stat-value" id="online-players">1,247</div>
            </div>
            <div class="stat-box">
                <div class="stat-label">شرط کل این دور</div>
                <div class="stat-value" id="total-bet">24,850</div>
                <div class="stat-label">سکه</div>
            </div>
            <div class="stat-box">
                <div class="stat-label">بالاترین ضریب</div>
                <div class="stat-value" id="highest-multiplier">18.75x</div>
            </div>
        </div>

        <div class="game-container">
            <div class="game-area">
                <div class="game-header">
                    <div class="game-timer">
                        <i class="fas fa-clock"></i>
                        <span id="next-game-timer">10</span>
                        <span>ثانیه تا شروع دور بعدی</span>
                    </div>
                    <div class="game-state" id="game-state">در حال پردازش...</div>
                </div>
                
                <div class="multiplier-display" id="multiplier">1.00x</div>
                
                <div class="rocket-animation" id="rocket">
                    <i class="fas fa-rocket"></i>
                </div>
                
                <div class="graph-container">
                    <canvas id="game-graph"></canvas>
                </div>

                <div class="game-controls">
                    <div class="bet-input-container">
                        <input type="number" class="bet-input" id="bet-amount" placeholder="مبلغ شرط" value="100" min="10" max="10000">
                        <div class="quick-bets">
                            <button class="quick-bet-btn" data-value="10">10</button>
                            <button class="quick-bet-btn" data-value="50">50</button>
                            <button class="quick-bet-btn" data-value="100">100</button>
                            <button class="quick-bet-btn" data-value="500">500</button>
                            <button class="quick-bet-btn" data-value="1000">1,000</button>
                        </div>
                    </div>
                    <button class="btn btn-primary" id="place-bet">
                        <i class="fas fa-play"></i>
                        <span>شروع بازی</span>
                    </button>
                    <button class="btn btn-secondary" id="cash-out" disabled>
                        <i class="fas fa-hand-holding-usd"></i>
                        <span>نقد کردن</span>
                    </button>
                </div>

                <div class="auto-cashout">
                    <span>نقد کردن خودکار در:</span>
                    <input type="number" id="auto-cashout" value="2.00" step="0.1" min="1.1" max="100">
                    <span>X</span>
                    <button class="btn btn-primary" id="set-auto-cashout">
                        <i class="fas fa-check"></i>
                        <span>فعال‌سازی</span>
                    </button>
                </div>

                <div class="profit-calculator">
                    <div>سود احتمالی: <span id="profit-multiplier">2.00x</span></div>
                    <div class="profit-display" id="profit-amount">200 سکه</div>
                </div>
            </div>

            <div class="side-panel">
                <div class="panel">
                    <div class="panel-title">
                        <span>تاریخچه بازی‌ها</span>
                        <i class="fas fa-history"></i>
                    </div>
                    <div class="history-list" id="history-list">
                        <div class="history-item">
                            <span class="time">۲ ثانیه پیش</span>
                            <span class="multiplier high">۸.۴۵x</span>
                        </div>
                        <div class="history-item">
                            <span class="time">۱۵ ثانیه پیش</span>
                            <span class="multiplier low">۱.۲۳x</span>
                        </div>
                        <div class="history-item">
                            <span class="time">۴۵ ثانیه پیش</span>
                            <span class="multiplier medium">۳.۷۶x</span>
                        </div>
                        <div class="history-item">
                            <span class="time">۱ دقیقه پیش</span>
                            <span class="multiplier high">۵.۹۸x</span>
                        </div>
                        <div class="history-item">
                            <span class="time">۲ دقیقه پیش</span>
                            <span class="multiplier low">۱.۰۵x</span>
                        </div>
                    </div>
                </div>

                <div class="panel">
                    <div class="panel-title">
                        <span>شرط‌های اخیر</span>
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="bets-list" id="bets-list">
                        <div class="bet-item">
                            <div class="user-bet">
                                <div class="avatar">U1</div>
                                <span>User1</span>
                            </div>
                            <div class="bet-amount">100 (۲.۳۴x)</div>
                        </div>
                        <div class="bet-item">
                            <div class="user-bet">
                                <div class="avatar">U2</div>
                                <span>User2</span>
                            </div>
                            <div class="bet-amount">50 (۱.۸۹x)</div>
                        </div>
                        <div class="bet-item">
                            <div class="user-bet">
                                <div class="avatar">U3</div>
                                <span>User3</span>
                            </div>
                            <div class="bet-amount">200 (۵.۰۱x)</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // تنظیم تم‌ها
            const themeButtons = document.querySelectorAll('.theme-btn');
            themeButtons.forEach(btn => {
                btn.addEventListener('click', function() {
                    const theme = this.getAttribute('data-theme');
                    document.body.className = 'theme-' + theme;
                    
                    themeButtons.forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                });
            });

            // متغیرهای بازی
            let balance = 5000;
            let currentBet = 0;
            let currentMultiplier = 1.0;
            let gameRunning = false;
            let cashoutAt = 2.0;
            let autoCashoutEnabled = false;
            let gameInterval;
            let crashPoint = 0;
            let gameHistory = [];
            let nextGameTimer = 10;

            // عناصر DOM
            const balanceEl = document.getElementById('balance');
            const multiplierEl = document.getElementById('multiplier');
            const betAmountEl = document.getElementById('bet-amount');
            const placeBetBtn = document.getElementById('place-bet');
            const cashOutBtn = document.getElementById('cash-out');
            const autoCashoutEl = document.getElementById('auto-cashout');
            const setAutoCashoutBtn = document.getElementById('set-auto-cashout');
            const gameStateEl = document.getElementById('game-state');
            const nextGameTimerEl = document.getElementById('next-game-timer');
            const rocketEl = document.getElementById('rocket');
            const canvas = document.getElementById('game-graph');
            const ctx = canvas.getContext('2d');
            const profitMultiplierEl = document.getElementById('profit-multiplier');
            const profitAmountEl = document.getElementById('profit-amount');
            const onlinePlayersEl = document.getElementById('online-players');
            const totalBetEl = document.getElementById('total-bet');
            const highestMultiplierEl = document.getElementById('highest-multiplier');

            // محاسبه نقطه Crash با الگوریتمی که به نفع خانه است
            function calculateCrashPoint() {
                // الگوریتمی که احتمال Crash در اعداد پایین را بیشتر می‌کند
                const houseEdge = 0.01; // 1% advantage for house
                const e = 1 - houseEdge;
                const random = Math.random();
                
                // فرمول محاسبه نقطه Crash
                return Math.max(1, (e / (1 - random)) * 0.97);
            }

            // اندازه‌گیری Canvas
            function resizeCanvas() {
                canvas.width = canvas.offsetWidth;
                canvas.height = canvas.offsetHeight;
                drawGraph();
            }

            // رسم نمودار
            function drawGraph() {
                ctx.clearRect(0, 0, canvas.width, canvas.height);
                
                // رسم شبکه
                ctx.strokeStyle = 'rgba(255, 255, 255, 0.1)';
                ctx.lineWidth = 1;
                
                // خطوط عمودی
                for (let x = 0; x < canvas.width; x += 50) {
                    ctx.beginPath();
                    ctx.moveTo(x, 0);
                    ctx.lineTo(x, canvas.height);
                    ctx.stroke();
                }
                
                // خطوط افقی
                for (let y = 0; y < canvas.height; y += 50) {
                    ctx.beginPath();
                    ctx.moveTo(0, y);
                    ctx.lineTo(canvas.width, y);
                    ctx.stroke();
                }
                
                if (gameRunning) {
                    // رسم خط ضریب
                    ctx.beginPath();
                    ctx.moveTo(0, canvas.height - 50);
                    
                    const progress = Math.min(currentMultiplier / crashPoint, 1);
                    const x = progress * (canvas.width - 100);
                    const y = canvas.height - 50 - (currentMultiplier * 20);
                    
                    ctx.lineTo(x, y);
                    ctx.strokeStyle = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                    ctx.lineWidth = 3;
                    ctx.stroke();
                    
                    // نقطه روی نمودار
                    ctx.beginPath();
                    ctx.arc(x, y, 6, 0, Math.PI * 2);
                    ctx.fillStyle = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                    ctx.fill();

                    // نمایش نقطه Crash
                    if (crashPoint > 0) {
                        const crashY = canvas.height - 50 - (crashPoint * 20);
                        ctx.setLineDash([5, 3]);
                        ctx.beginPath();
                        ctx.moveTo(0, crashY);
                        ctx.lineTo(canvas.width, crashY);
                        ctx.strokeStyle = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                        ctx.lineWidth = 2;
                        ctx.stroke();
                        ctx.setLineDash([]);

                        ctx.fillStyle = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                        ctx.fillText(`Crash: ${crashPoint.toFixed(2)}x`, canvas.width - 100, crashY - 10);
                    }
                }
            }

            // به‌روزرسانی ماشین حساب سود
            function updateProfitCalculator() {
                const betAmount = parseInt(betAmountEl.value) || 0;
                const multiplier = parseFloat(autoCashoutEl.value) || 2.0;
                const profit = Math.floor(betAmount * multiplier);
                
                profitMultiplierEl.textContent = multiplier.toFixed(2) + 'x';
                profitAmountEl.textContent = profit.toLocaleString() + ' سکه';
            }

            // شروع بازی
            function startGame() {
                if (currentBet <= 0) return;
                
                crashPoint = calculateCrashPoint();
                gameRunning = true;
                currentMultiplier = 1.0;
                placeBetBtn.disabled = true;
                cashOutBtn.disabled = false;
                gameStateEl.textContent = "بازی در حال انجام...";
                rocketEl.classList.add('flying');
                
                // کاهش موجودی با میزان شرط
                balance -= currentBet;
                balanceEl.textContent = balance.toLocaleString();
                
                // شبیه‌سازی افزایش ضریب
                gameInterval = setInterval(() => {
                    // افزایش ضریب
                    const increment = 0.05 + (Math.random() * 0.1);
                    currentMultiplier += increment;
                    
                    multiplierEl.textContent = currentMultiplier.toFixed(2) + "x";
                    
                    // به‌روزرسانی موقعیت موشک
                    const progress = Math.min(currentMultiplier / crashPoint, 1);
                    rocketEl.style.bottom = 70 - (60 * progress) + '%';
                    
                    // بررسی نقد کردن خودکار
                    if (autoCashoutEnabled && currentMultiplier >= cashoutAt) {
                        cashOut();
                        return;
                    }
                    
                    // بررسی crash
                    if (currentMultiplier >= crashPoint) {
                        endGame();
                        return;
                    }
                    
                    drawGraph();
                }, 100);
            }

            // نقد کردن
            function cashOut() {
                if (!gameRunning) return;
                
                const winAmount = Math.floor(currentBet * currentMultiplier);
                balance += winAmount;
                balanceEl.textContent = balance.toLocaleString();
                
                endGame();
                
                // نمایش پیام سود
                gameStateEl.textContent = `نقد کردن موفق! شما ${winAmount.toLocaleString()} سکه برنده شدید!`;
                gameStateEl.style.color = getComputedStyle(document.documentElement).getPropertyValue('--secondary');
                
                setTimeout(() => {
                    gameStateEl.textContent = "در حال پردازش...";
                    gameStateEl.style.color = getComputedStyle(document.documentElement).getPropertyValue('--primary');
                }, 3000);
            }

            // پایان بازی
            function endGame() {
                clearInterval(gameInterval);
                gameRunning = false;
                placeBetBtn.disabled = false;
                cashOutBtn.disabled = true;
                rocketEl.classList.remove('flying');
                
                // افزودن به تاریخچه
                addToHistory(currentMultiplier);
                
                // بازنشانی
                currentBet = 0;
                betAmountEl.value = "100";
                
                drawGraph();
                
                // شروع تایمر برای دور بعدی
                nextGameTimer = 10;
                const timerInterval = setInterval(() => {
                    nextGameTimer--;
                    nextGameTimerEl.textContent = nextGameTimer;
                    
                    if (nextGameTimer <= 0) {
                        clearInterval(timerInterval);
                        gameStateEl.textContent = "آماده برای شرط‌بندی";
                    }
                }, 1000);
            }

            // افزودن به تاریخچه
            function addToHistory(multiplier) {
                const historyList = document.getElementById('history-list');
                const historyItem = document.createElement('div');
                historyItem.className = 'history-item';
                
                const timeSpan = document.createElement('span');
                timeSpan.className = 'time';
                timeSpan.textContent = 'هم اکنون';
                
                const multiplierSpan = document.createElement('span');
                multiplierSpan.className = 'multiplier';
                
                if (multiplier >= 5) multiplierSpan.classList.add('high');
                else if (multiplier >= 2) multiplierSpan.classList.add('medium');
                else multiplierSpan.classList.add('low');
                
                multiplierSpan.textContent = multiplier.toFixed(2) + 'x';
                
                historyItem.appendChild(timeSpan);
                historyItem.appendChild(multiplierSpan);
                
                historyList.insertBefore(historyItem, historyList.firstChild);
                
                // حذف آیتم‌های اضافی
                if (historyList.children.length > 10) {
                    historyList.removeChild(historyList.lastChild);
                }
                
                // به‌روزرسانی بالاترین ضریب
                const currentHighest = parseFloat(highestMultiplierEl.textContent);
                if (multiplier > currentHighest) {
                    highestMultiplierEl.textContent = multiplier.toFixed(2) + 'x';
                }
                
                // ذخیره در تاریخچه بازی
                gameHistory.push(multiplier);
            }

            // رویدادهای کلیک
            placeBetBtn.addEventListener('click', function() {
                const betAmount = parseInt(betAmountEl.value);
                
                if (isNaN(betAmount) || betAmount < 10) {
                    alert("لطفا مبلغ شرط معتبر وارد کنید (حداقل 10)");
                    return;
                }
                
                if (betAmount > balance) {
                    alert("موجودی شما کافی نیست!");
                    return;
                }
                
                currentBet = betAmount;
                startGame();
            });

            cashOutBtn.addEventListener('click', cashOut);

            setAutoCashoutBtn.addEventListener('click', function() {
                cashoutAt = parseFloat(autoCashoutEl.value) || 2.0;
                autoCashoutEnabled = true;
                alert(`نقد کردن خودکار در ${cashoutAt.toFixed(2)}x فعال شد`);
            });

            // دکمه‌های شرط سریع
            document.querySelectorAll('.quick-bet-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const value = parseInt(this.getAttribute('data-value'));
                    betAmountEl.value = value;
                    updateProfitCalculator();
                });
            });

            // تغییرات در inputها
            betAmountEl.addEventListener('input', updateProfitCalculator);
            autoCashoutEl.addEventListener('input', updateProfitCalculator);

            // تنظیم خودکار اندازه Canvas
            window.addEventListener('resize', resizeCanvas);
            resizeCanvas();
            drawGraph();
            updateProfitCalculator();

            // شبیه‌سازی بازیکنان آنلاین و شرط‌ها
            setInterval(() => {
                // بازیکنان آنلاین
                const onlinePlayers = parseInt(onlinePlayersEl.textContent.replace(/,/g, ''));
                const change = Math.floor(Math.random() * 10) - 3;
                let newOnline = Math.max(1000, onlinePlayers + change);
                onlinePlayersEl.textContent = newOnline.toLocaleString();
                
                // شرط کل
                const totalBet = parseInt(totalBetEl.textContent.replace(/,/g, ''));
                const betChange = Math.floor(Math.random() * 500) - 100;
                let newTotalBet = Math.max(1000, totalBet + betChange);
                totalBetEl.textContent = newTotalBet.toLocaleString();
            }, 5000);
        });
    </script>
</body>
</html>